package ru.yandex.direct.core.entity.strategy.service

import org.jooq.DSLContext
import org.springframework.stereotype.Service
import ru.yandex.direct.autobudget.restart.model.PackageStrategyDto
import ru.yandex.direct.autobudget.restart.model.PackageStrategyRestartResult
import ru.yandex.direct.autobudget.restart.repository.CampRestartData
import ru.yandex.direct.autobudget.restart.repository.CampaignAutobudgetRestartRepository
import ru.yandex.direct.autobudget.restart.repository.RestartTimes
import ru.yandex.direct.autobudget.restart.repository.StrategyRestartData
import ru.yandex.direct.autobudget.restart.service.AutobudgetRestartService
import ru.yandex.direct.autobudget.restart.service.AutobudgetStrategyRestartService
import ru.yandex.direct.autobudget.restart.service.InitialRestartsProvider
import ru.yandex.direct.autobudget.restart.service.PackageStrategyRestartData
import ru.yandex.direct.core.entity.campaign.CampaignAutobudgetRestartUtils
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.WalletHasMoneyChecker
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpa
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpaPerFilter
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpcPerFilter
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpi
import ru.yandex.direct.core.entity.strategy.model.AutobudgetCrr
import ru.yandex.direct.core.entity.strategy.model.AutobudgetMedia
import ru.yandex.direct.core.entity.strategy.model.AutobudgetRoi
import ru.yandex.direct.core.entity.strategy.model.AutobudgetWeekBundle
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.model.DefaultManualStrategy
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgBid
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgCpm
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgCpv
import ru.yandex.direct.core.entity.strategy.model.StrategyWithBid
import ru.yandex.direct.core.entity.strategy.model.StrategyWithConversion
import ru.yandex.direct.core.entity.strategy.model.StrategyWithCustomPeriodBudget
import ru.yandex.direct.core.entity.strategy.model.StrategyWithDayBudget
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMeaningfulGoals
import ru.yandex.direct.core.entity.strategy.model.StrategyWithPayForConversion
import ru.yandex.direct.core.entity.strategy.model.StrategyWithWeeklyBudget
import ru.yandex.direct.dbutil.wrapper.DslContextProvider

/**
 * Сервис для вычисления рестарта автобюджета публичных пакетных стратегий.
 *
 * */
@Service
class PublicPackageStrategyAutobudgetRestartService(
    private val autobudgetRestartService: AutobudgetStrategyRestartService,
    private val campAutobudgetRestartService: AutobudgetRestartService,
    private val campaignAutobudgetRestartRepository: CampaignAutobudgetRestartRepository,
    private val hasMoneyChecker: WalletHasMoneyChecker,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val metrikaCampaignRepository: MetrikaCampaignRepository,
    private val dslContextProvider: DslContextProvider
) {

    fun recalculateAndSaveRestarts(shard: Int, strategies: List<CommonStrategy>): List<PackageStrategyRestartResult> {
        if (strategies.isEmpty()) {
            return emptyList()
        }
        require(strategies.find { !it.isPublic } == null) { "All strategies should be public" }
        val dtos = getRestartDto(shard, strategies)
        val initialRestartsProvider = initialRestartsProvider(shard, strategies)
        var result = emptyList<PackageStrategyRestartResult>()
        dslContextProvider.ppcTransaction(shard) { configuration ->
            val context = configuration.dsl()
            result = autobudgetRestartService.recalculateAndSaveRestarts(context, dtos, initialRestartsProvider, true)
            spreadRestartsToCampaigns(
                context,
                result,
                dtos,
                strategies.associateBy(CommonStrategy::getId) { it.cids ?: emptyList() }
            )
        }
        return result
    }

    private fun spreadRestartsToCampaigns(
        dslContext: DSLContext,
        strategyRestarts: List<PackageStrategyRestartResult>,
        strategyDtos: List<PackageStrategyRestartData>,
        strategyIdToCampaignId: Map<Long, List<Long>>
    ) {
        val dtoByStrategyId = strategyDtos.associateBy { it.strategyId }
        val restartByStrategyId = strategyRestarts.associateBy { it.strategyId }
        val restartsByCampaignId = strategyIdToCampaignId.flatMap { (strategyId, cids) ->
            cids.mapNotNull { cid ->
                restartByStrategyId[strategyId]?.let { cid to it }
            }
        }.toMap()
        val campaigns = campaignTypedRepository.getTypedCampaigns(dslContext, restartsByCampaignId.keys)
            .mapNotNull { it as? CommonCampaign }
            .associateBy { it.id }
        val campaignIdsWithCombinedGoals =
            metrikaCampaignRepository.getCampaignIdsWithCombinedGoals(dslContext, campaigns.keys)
        val campRestarts = restartsByCampaignId.mapKeys { (cid, _) -> campaigns[cid] }
            .mapNotNull { (campaign, strategyRestart) ->
                campaign?.let {
                    val dto = CampaignAutobudgetRestartUtils.getStrategyDto(
                        it,
                        dtoByStrategyId[strategyRestart.strategyId]?.dto?.hasMoney ?: false,
                        campaignIdsWithCombinedGoals.contains(it.id)
                    )
                    CampRestartData(
                        it.id,
                        it.orderId,
                        dto,
                        RestartTimes(
                            strategyRestart.restartTime,
                            strategyRestart.softRestartTime,
                            strategyRestart.restartReason,
                        ),
                        strategyRestart.state
                    )
                }
            }
        campaignAutobudgetRestartRepository.saveAutobudgetRestartData(
            dslContext,
            campRestarts
        )
    }

    private fun getRestartDto(
        shard: Int,
        strategies: List<CommonStrategy>
    ): List<PackageStrategyRestartData> {
        val wallets =
            campaignTypedRepository.getSafely(shard, strategies.map { it.walletId }, WalletTypedCampaign::class.java)
        val hasMoneyByStrategyId = hasMoneyChecker.calcHasMoneyForStrategies(
            shard,
            strategies,
            wallets
        )
        return strategies.map {
            PackageStrategyRestartData(it.id, toStrategyDto(hasMoneyByStrategyId[it.id]!!, it))
        }
    }

    private fun initialRestartsProvider(shard: Int, strategies: List<CommonStrategy>): InitialRestartsProvider =
        InitialRestartsProvider { initialRestarts(shard, strategies) }

    private fun initialRestarts(shard: Int, strategies: List<CommonStrategy>): Map<Long, RestartTimes?> {
        val cidsByStrategyId = strategies.associateBy({ it.id }) { it.cids ?: emptyList() }
        val campaigns = campaignTypedRepository.getTyped(shard, cidsByStrategyId.values.flatten())
            .mapNotNull { it as? CampaignWithPackageStrategy }
        val wallets =
            campaignTypedRepository.getSafely(shard, campaigns.map { it.walletId }, WalletTypedCampaign::class.java)
        val hasMoneyByCid = hasMoneyChecker.calcHasMoney(
            shard,
            campaigns,
            wallets
        )
        val campaignIdsWithCombinedGoals =
            metrikaCampaignRepository.getCampaignIdsWithCombinedGoals(shard, campaigns.map { it.id })
        val dtoByCampaignId = campaigns.associate {
            val dto = CampaignAutobudgetRestartUtils.getStrategyDto(
                it,
                hasMoney = hasMoneyByCid[it.id]!!,
                campaignIdsWithCombinedGoals.contains(it.id)
            )
            it.id to dto
        }
        val campRestarts = campAutobudgetRestartService.getActualRestartTimes(shard, dtoByCampaignId)
            .associateBy { it.cid }
        return cidsByStrategyId.mapValues { (_, cids) ->
            cids.mapNotNull(campRestarts::get)
                .minByOrNull { it.restartTime }
                ?.let {
                    RestartTimes(
                        it.restartTime,
                        it.softRestartTime,
                        it.restartReason
                    )
                }
        }
    }

    companion object {
        fun toStrategyDto(hasMoney: Boolean, strategy: CommonStrategy): PackageStrategyDto {
            return PackageStrategyDto(
                hasMoney,
                strategy.type,
                strategy.isPublic,
                strategy.attributionModel,
                (strategy as? StrategyWithWeeklyBudget)?.sum,
                (strategy as? StrategyWithMeaningfulGoals)?.meaningfulGoals,
                (strategy as? StrategyWithBid)?.bid,
                (strategy as? StrategyWithPayForConversion)?.isPayForConversionEnabled,
                (strategy as? AutobudgetAvgCpaPerFilter)?.filterAvgCpa,
                (strategy as? AutobudgetAvgCpcPerFilter)?.filterAvgBid,
                (strategy as? StrategyWithCustomPeriodBudget)?.start,
                (strategy as? StrategyWithCustomPeriodBudget)?.finish,
                (strategy as? StrategyWithCustomPeriodBudget)?.autoProlongation,
                (strategy as? StrategyWithCustomPeriodBudget)?.budget,
                (strategy as? AutobudgetCrr)?.crr,
                (strategy as? AutobudgetMedia)?.date,
                (strategy as? AutobudgetRoi)?.reserveReturn,
                (strategy as? AutobudgetRoi)?.profitability,
                (strategy as? AutobudgetRoi)?.roiCoef,
                (strategy as? StrategyWithDayBudget)?.dayBudgetShowMode,
                (strategy as? StrategyWithDayBudget)?.dayBudget,
                (strategy as? AutobudgetAvgCpi)?.avgCpi,
                (strategy as? StrategyWithAvgCpv)?.avgCpv,
                (strategy as? AutobudgetAvgCpa)?.avgCpa,
                (strategy as? StrategyWithConversion)?.goalId,
                (strategy as? StrategyWithAvgCpm)?.avgCpm,
                (strategy as? AutobudgetWeekBundle)?.limitClicks,
                (strategy as? DefaultManualStrategy)?.enableCpcHold,
                (strategy as? StrategyWithAvgBid)?.avgBid
            )
        }

        fun buildStrategyRestartData(
            campRestartData: CampRestartData,
            commonStrategy: CommonStrategy?
        ): StrategyRestartData? =
            commonStrategy?.let {
                val dto = toStrategyDto(campRestartData.strategyData.hasMoney ?: false, it)
                StrategyRestartData(
                    commonStrategy.id,
                    dto,
                    campRestartData.times,
                    campRestartData.state
                )
            }
    }
}
