package ru.yandex.direct.core.entity.strategy.service.add;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.metrika.service.commongoals.CommonCountersService;
import ru.yandex.direct.core.entity.strategy.container.StrategyAddOperationContainer;
import ru.yandex.direct.core.entity.strategy.container.StrategyRepositoryContainer;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMetrikaCounters;
import ru.yandex.direct.core.entity.strategy.repository.StrategyModifyRepository;
import ru.yandex.direct.core.entity.strategy.service.CampaignUpdateHelper;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;

import static ru.yandex.direct.utils.FunctionalUtils.selectList;

@Service
@ParametersAreNonnullByDefault
public class StrategyAddOperationService {
    private final CommonCountersService commonCountersService;
    private final DslContextProvider ppcDslContextProvider;

    private final StrategyModifyRepository strategyModifyRepository;
    private final StrategyAddOperationTypeSupportFacade addOperationTypeSupportFacade;
    private final CampaignUpdateHelper campaignUpdateHelper;

    @Autowired
    public StrategyAddOperationService(CommonCountersService commonCountersService,
                                       DslContextProvider ppcDslContextProvider,
                                       StrategyModifyRepository strategyModifyRepository,
                                       StrategyAddOperationTypeSupportFacade addOperationTypeSupportFacade,
                                       CampaignUpdateHelper campaignUpdateHelper) {
        this.commonCountersService = commonCountersService;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.strategyModifyRepository = strategyModifyRepository;
        this.addOperationTypeSupportFacade = addOperationTypeSupportFacade;
        this.campaignUpdateHelper = campaignUpdateHelper;
    }

    public List<Long> execute(StrategyAddOperationContainer operationContainer, List<BaseStrategy> models) {
        var repositoryContainer = createRepositoryContainer(
                operationContainer.getShard(),
                operationContainer.getClientId(),
                models,
                operationContainer.getOptions().isStrategyIdsAlreadyFilled(),
                operationContainer.getOptions().isCampaignToPackageStrategyOneshot());

        List<Long> addedStrategyIds = new ArrayList<>(models.size());
        ppcDslContextProvider.ppcTransaction(repositoryContainer.getShard(), conf -> {
            DSLContext dsl = conf.dsl();
            addedStrategyIds.addAll(execute(dsl, repositoryContainer, models, operationContainer));
        });
        var commonStrategies = selectList(models, CommonStrategy.class);
        campaignUpdateHelper.updateCampaignsOutOfTransaction(operationContainer, commonStrategies);
        return addedStrategyIds;
    }

    public List<Long> execute(DSLContext dsl, StrategyAddOperationContainer operationContainer,
                              List<BaseStrategy> models) {
        var repositoryContainer = createRepositoryContainer(
                operationContainer.getShard(),
                operationContainer.getClientId(),
                models,
                operationContainer.getOptions().isStrategyIdsAlreadyFilled(),
                operationContainer.getOptions().isCampaignToPackageStrategyOneshot());


        return execute(dsl, repositoryContainer, models, operationContainer);
    }

    private List<Long> execute(
            DSLContext dsl,
            StrategyRepositoryContainer repositoryContainer,
            List<BaseStrategy> models, StrategyAddOperationContainer operationContainer) {
        var addedStrategyIds = strategyModifyRepository.add(dsl, repositoryContainer, models);
        addOperationTypeSupportFacade.updateRelatedEntitiesInTransaction(dsl, operationContainer,
                models);
        return addedStrategyIds;
    }

    private StrategyRepositoryContainer createRepositoryContainer(
            int shard, ClientId clientId, List<BaseStrategy> validModelsToApply, boolean isStrategyIdsAlreadyFilled,
            boolean isCampaignToStrategyMigrationOneshot) {
        List<Long> counterIds = StreamEx.of(validModelsToApply)
                .select(StrategyWithMetrikaCounters.class)
                .flatCollection(StrategyWithMetrikaCounters::getMetrikaCounters)
                .distinct()
                .toList();
        List<UserCountersExtended> availableCountersFromMetrika = List.of();
        if (!isCampaignToStrategyMigrationOneshot) {
            availableCountersFromMetrika =
                    commonCountersService.getAvailableCountersFromMetrika(clientId, counterIds);
        }

        var counterInfoById = StreamEx.of(availableCountersFromMetrika)
                .flatCollection(UserCountersExtended::getCounters)
                .mapToEntry(CounterInfoDirect::getId, Function.identity())
                .distinctKeys()
                .mapKeys(Integer::longValue)
                .toMap();

        return new StrategyRepositoryContainer(shard, clientId, counterInfoById, isStrategyIdsAlreadyFilled);
    }
}
