package ru.yandex.direct.core.entity.strategy.service.update;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.strategy.container.StrategyOperationOptions;
import ru.yandex.direct.core.entity.strategy.container.StrategyUpdateOperationContainer;
import ru.yandex.direct.core.entity.strategy.container.StrategyUpdateOperationContainerService;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy;
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository;
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants;
import ru.yandex.direct.core.entity.strategy.validation.update.StrategyUpdateValidationTypeSupportFacade;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ChangesAppliedStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.operation.update.ModelStubProvider.getModelStub;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class StrategyUpdateOperation<T extends BaseStrategy> extends SimpleAbstractUpdateOperation<T, Long> {

    private final int shard;

    private final StrategyUpdateOperationContainerService strategyUpdateOperationContainerService;
    private final StrategyUpdateValidationTypeSupportFacade validationTypeSupportFacade;
    private final StrategyUpdateOperationTypeSupportFacade updateOperationTypeSupportFacade;
    private final StrategyTypedRepository strategyTypedRepository;
    private final StrategyUpdateOperationService strategyUpdateOperationService;

    private final ModelChangesValidationTool updateValidationTool;

    private final StrategyUpdateOperationContainer operationContainer;

    //вытаскиваем сюда, чтобы не менять пока абстрактную операцию
    private Map<Long, T> currentModels;


    public StrategyUpdateOperation(int shard,
                                   ClientId clientId,
                                   Long clientUid,
                                   Long operator,
                                   StrategyOperationOptions operationOptions,
                                   List<ModelChanges<T>> modelChanges,
                                   StrategyUpdateOperationContainerService strategyUpdateOperationContainerService,
                                   StrategyUpdateOperationService strategyUpdateOperationService,
                                   StrategyUpdateValidationTypeSupportFacade validationTypeSupportFacade,
                                   StrategyUpdateOperationTypeSupportFacade updateOperationTypeSupportFacade,
                                   StrategyTypedRepository strategyTypedRepository,
                                   Class<T> clazz) {
        super(Applicability.PARTIAL, modelChanges, getModelStub(clazz,
                StrategyConstants.STRATEGY_CLASS_TO_TYPE.keySet()));

        this.shard = shard;

        this.strategyUpdateOperationContainerService = strategyUpdateOperationContainerService;
        this.strategyUpdateOperationService = strategyUpdateOperationService;
        this.validationTypeSupportFacade = validationTypeSupportFacade;
        this.updateOperationTypeSupportFacade = updateOperationTypeSupportFacade;
        this.strategyTypedRepository = strategyTypedRepository;
        this.operationContainer = new StrategyUpdateOperationContainer(shard, clientId, clientUid, operator, operationOptions);
        this.updateValidationTool = ModelChangesValidationTool.builder().build();
    }

    @SuppressWarnings("rawtypes")
    @Override
    protected ValidationResult<List<ModelChanges<T>>, Defect> validateModelChanges(List<ModelChanges<T>> modelChanges) {
        var vr = new ValidationResult<>(modelChanges, Defect.class);

        var strategies = strategyTypedRepository.getStrictlyFullyFilled(
                shard,
                mapList(modelChanges, ModelChanges::getId),
                CommonStrategy.class);
        var clientStrategies = filterList(strategies,
                strategy -> Objects.equals(operationContainer.getClientId().asLong(), strategy.getClientId()));
        strategyUpdateOperationContainerService.fillContainers(operationContainer, clientStrategies, modelChanges, null);
        updateValidationTool.validateModelChangesList(vr, operationContainer::getStrategyIds);

        if (vr.hasAnyErrors()) {
            return vr;
        }

        validationTypeSupportFacade.preValidate(operationContainer, vr);
        return vr;
    }

    @Override
    protected Map<Integer, AppliedChanges<T>> getAppliedChangesForValidModelChanges(
            Map<Integer, ModelChanges<T>> validModelChanges,
            Map<Long, T> models) {

        this.currentModels = models;
        //для случая с изменением типа отдельная логика
        return strategyUpdateOperationService.getAppliedChangesById(Map.of(), Set.of(), validModelChanges, models);
    }

    @Override
    protected void onChangesApplied(ChangesAppliedStep<T> changesAppliedStep) {
        List<AppliedChanges<T>> validAppliedChanges =
                new ArrayList<>(changesAppliedStep.getAppliedChangesForValidModelChanges());
        updateOperationTypeSupportFacade.onChangesApplied(operationContainer, validAppliedChanges);
    }

    @Override
    protected ValidationResult<List<ModelChanges<T>>, Defect> validateModelChangesBeforeApply(ValidationResult<List<ModelChanges<T>>, Defect> preValidateResult, Map<Long, T> models) {
        validationTypeSupportFacade.validateBeforeApply(operationContainer, preValidateResult, models);
        return preValidateResult;
    }

    @Override
    protected ValidationResult<List<T>, Defect> validateAppliedChanges(
            ValidationResult<List<T>, Defect> validationResult,
            Map<Integer, AppliedChanges<T>> appliedChangesForValidModelChanges) {
        validationTypeSupportFacade.validate(operationContainer, validationResult,
                appliedChangesForValidModelChanges);
        return validationResult;
    }

    @Override
    protected void onAppliedChangesValidated(
            AppliedChangesValidatedStep<T> appliedChangesValidatedStep) {
        var validAppliedChanges = new ArrayList<>(appliedChangesValidatedStep.getValidAppliedChanges());

        updateOperationTypeSupportFacade.onAppliedChangesValidated(operationContainer, validAppliedChanges);
    }


    //тут странное с типами -- почему-то делаем операцию принимающую список<T>
    //хотя на самом деле могут приходить все типы, вероятно предполагается некий Common
    @Override
    protected Collection<T> getModels(Collection<Long> ids) {
        //noinspection unchecked
        return (Collection<T>) strategyTypedRepository.getTyped(
                shard,
                ids);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<T>> applicableAppliedChanges) {
        return strategyUpdateOperationService.execute(
                operationContainer,
                getValidModelChanges(),
                currentModels,
                applicableAppliedChanges);
    }
}
