package ru.yandex.direct.core.entity.strategy.service.update;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.campaign.service.type.RestrictedCampaignsOperationContainer;
import ru.yandex.direct.core.entity.strategy.container.StrategyAdditionalActionsContainer;
import ru.yandex.direct.core.entity.strategy.container.StrategyAdditionalActionsService;
import ru.yandex.direct.core.entity.strategy.container.StrategyOperationOptions;
import ru.yandex.direct.core.entity.strategy.container.StrategyRepositoryContainer;
import ru.yandex.direct.core.entity.strategy.container.StrategyUpdateOperationContainer;
import ru.yandex.direct.core.entity.strategy.container.StrategyUpdateOperationContainerService;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyName;
import ru.yandex.direct.core.entity.strategy.repository.StrategyModifyRepository;
import ru.yandex.direct.core.entity.strategy.service.CampaignUpdateHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;

import static ru.yandex.direct.common.db.PpcPropertyNames.MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.PROPERTIES_BY_TYPE;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.TYPE_TO_STRATEGY_CLASS_SUPPLIER;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class StrategyUpdateOperationService {
    private final StrategyModifyRepository strategyModifyRepository;
    private final StrategyUpdateOperationTypeSupportFacade updateOperationTypeSupportFacade;
    private final StrategyAdditionalActionsService strategyAdditionalActionsService;
    private final StrategyUpdateOperationContainerService strategyUpdateOperationContainerService;
    private final CampaignUpdateHelper campaignUpdateHelper;
    private final PpcProperty<Integer> maxCampaignsAllowedToLinkToStrategyProperty;

    private final DslContextProvider ppcDslContextProvider;

    @Autowired
    public StrategyUpdateOperationService(StrategyModifyRepository strategyModifyRepository,
                                          StrategyUpdateOperationTypeSupportFacade updateOperationTypeSupportFacade,
                                          StrategyAdditionalActionsService strategyAdditionalActionsService,
                                          StrategyUpdateOperationContainerService strategyUpdateOperationContainerService,
                                          CampaignUpdateHelper campaignUpdateHelper,
                                          DslContextProvider ppcDslContextProvider,
                                          PpcPropertiesSupport ppcPropertiesSupport) {
        this.strategyModifyRepository = strategyModifyRepository;
        this.updateOperationTypeSupportFacade = updateOperationTypeSupportFacade;
        this.strategyAdditionalActionsService = strategyAdditionalActionsService;
        this.strategyUpdateOperationContainerService = strategyUpdateOperationContainerService;
        this.campaignUpdateHelper = campaignUpdateHelper;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.maxCampaignsAllowedToLinkToStrategyProperty =
                ppcPropertiesSupport.get(MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY);
    }

    public <T extends BaseStrategy> List<Long> execute(
            DSLContext context,
            StrategyUpdateOperationContainer operationContainer,
            Map<Integer, ModelChanges<T>> validModelChanges,
            Map<Long, T> currentModels,
            List<AppliedChanges<T>> applicableAppliedChanges) {
        var repositoryContainer =
                strategyUpdateOperationContainerService.createRepositoryContainer(
                        operationContainer, applicableAppliedChanges,
                        operationContainer.getOptions().isCampaignToPackageStrategyOneshot());
        List<? extends AppliedChanges<? extends BaseStrategy>> appliedChangesWithPropertiesUnusedInNewType =
                calculateAppliedChangesWithPropertiesUnusedInNewType(operationContainer,
                        validModelChanges, currentModels);
        StrategyAdditionalActionsContainer additionalActionsContainer = new StrategyAdditionalActionsContainer();
        updateOperationTypeSupportFacade.addToAdditionalActionsContainer(additionalActionsContainer,
                operationContainer, applicableAppliedChanges);

        execute(context, operationContainer, applicableAppliedChanges, repositoryContainer,
                appliedChangesWithPropertiesUnusedInNewType, additionalActionsContainer);
        return mapList(applicableAppliedChanges, aac -> aac.getModel().getId());
    }

    public <T extends BaseStrategy> List<Long> execute(
            StrategyUpdateOperationContainer operationContainer,
            Map<Integer, ModelChanges<T>> validModelChanges,
            Map<Long, T> currentModels,
            List<AppliedChanges<T>> applicableAppliedChanges) {
        var repositoryContainer =
                strategyUpdateOperationContainerService.createRepositoryContainer(
                        operationContainer, applicableAppliedChanges,
                        operationContainer.getOptions().isCampaignToPackageStrategyOneshot());
        List<? extends AppliedChanges<? extends BaseStrategy>> appliedChangesWithPropertiesUnusedInNewType =
                calculateAppliedChangesWithPropertiesUnusedInNewType(operationContainer,
                        validModelChanges, currentModels);
        StrategyAdditionalActionsContainer additionalActionsContainer = new StrategyAdditionalActionsContainer();
        updateOperationTypeSupportFacade.addToAdditionalActionsContainer(additionalActionsContainer,
                operationContainer, applicableAppliedChanges);
        ppcDslContextProvider.ppcTransaction(operationContainer.getShard(), conf ->
                {
                    execute(conf.dsl(), operationContainer, applicableAppliedChanges, repositoryContainer,
                            appliedChangesWithPropertiesUnusedInNewType, additionalActionsContainer);
                }
        );
        campaignUpdateHelper.updateCampaignsOutOfTransaction(applicableAppliedChanges, operationContainer);
        return mapList(applicableAppliedChanges, aac -> aac.getModel().getId());
    }

    public <T extends BaseStrategy> void executeFromCampaign(DSLContext dslContext,
                                                             RestrictedCampaignsOperationContainer campaignContainer,
                                                             Map<Long, T> strategiesById,
                                                             List<ModelChanges<T>> strategiesModelChanges) {
        StrategyUpdateOperationContainer container =
                createStrategyUpdateOperationContainer(dslContext, campaignContainer, strategiesById,
                        strategiesModelChanges);
        var strategiesAppliedChangesById = getAppliedChangesById(Map.of(), Set.of(),
                EntryStream.of(strategiesModelChanges).toMap(), strategiesById);

        List<AppliedChanges<T>> strategiesAppliedChanges = EntryStream.of(strategiesAppliedChangesById)
                .sortedBy(Map.Entry::getKey)
                .values()
                .toList();
        execute(dslContext, container,
                EntryStream.of(strategiesModelChanges).toImmutableMap(),
                strategiesById, strategiesAppliedChanges);
    }

    private <T extends BaseStrategy> StrategyUpdateOperationContainer createStrategyUpdateOperationContainer(
            DSLContext dslContext,
            RestrictedCampaignsOperationContainer campaignContainer,
            Map<Long, T> strategyById,
            List<ModelChanges<T>> strategiesModelChanges) {
        StrategyOperationOptions options = new StrategyOperationOptions(
                false,
                false,
                false,
                false,
                true,
                false,
                true,
                maxCampaignsAllowedToLinkToStrategyProperty.getOrDefault(DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY));

        StrategyUpdateOperationContainer operationContainer =
                new StrategyUpdateOperationContainer(campaignContainer.getShard(), campaignContainer.getClientId(),
                        campaignContainer.getClientUid(), campaignContainer.getOperatorUid(), options);
        strategyUpdateOperationContainerService.fillContainers(operationContainer, strategyById.values(),
                strategiesModelChanges, dslContext);
        return operationContainer;
    }


    private <T extends BaseStrategy> void execute(
            DSLContext context,
            StrategyUpdateOperationContainer operationContainer,
            List<AppliedChanges<T>> applicableAppliedChanges,
            StrategyRepositoryContainer repositoryContainer,
            List<? extends AppliedChanges<? extends BaseStrategy>> appliedChangesWithPropertiesUnusedInNewType,
            StrategyAdditionalActionsContainer additionalActionsContainer) {
        //для стратегий этот этап может в итоге не понадобиться
        if (!appliedChangesWithPropertiesUnusedInNewType.isEmpty()) {
            strategyModifyRepository.update(context, repositoryContainer,
                    appliedChangesWithPropertiesUnusedInNewType);
        }
        //для стратегий этот этап может в итоге не понадобиться
        updateOperationTypeSupportFacade.updateRelatedEntitiesInTransaction(context, operationContainer,
                applicableAppliedChanges);
        strategyModifyRepository.update(context, repositoryContainer, applicableAppliedChanges);
        //для стратегий этот этап может в итоге не понадобиться
        strategyAdditionalActionsService.processAdditionalActionsContainer(
                context,
                additionalActionsContainer,
                operationContainer);
    }

    /**
     * Для того, чтобы удалить не имеющие смысла поля для нового типа, проходимся по интерфейсам старого типа,
     * неподходящим к новому -- и используем addModelChangesForCleaningUnsupportedTypeValues, если он реализован
     */
    public <T extends BaseStrategy> List<? extends AppliedChanges<? extends BaseStrategy>> calculateAppliedChangesWithPropertiesUnusedInNewType(
            StrategyUpdateOperationContainer operationContainer,
            Map<Integer, ModelChanges<T>> validModelChanges,
            Map<Long, T> models) {
        Map<Long, StrategyName> modelChangesWithChangedTypeById = EntryStream.of(validModelChanges)
                .filterValues(mc -> isTypeChanged(models, mc))
                .mapToKey((index, mc) -> mc.getId())
                .mapValues(mc -> mc.getChangedProp(BaseStrategy.TYPE))
                .toMap();
        List<? extends ModelChanges<? extends BaseStrategy>> modelChangesForCleaningTypeValues =
                updateOperationTypeSupportFacade.addModelChangesForCleaningUnsupportedTypeValues(operationContainer,
                        models.values(),
                        modelChangesWithChangedTypeById);

        //noinspection unchecked
        return mapList(modelChangesForCleaningTypeValues,
                mc -> ((ModelChanges<BaseStrategy>) mc).applyTo(models.get(mc.getId())));
    }

    public static <T extends BaseStrategy> boolean isTypeChanged(Map<Long, T> models, ModelChanges<T> mc) {
        return mc.isPropChanged(BaseStrategy.TYPE) && models.get(mc.getId()).getType() != mc.getChangedProp(BaseStrategy.TYPE);
    }

    public <T extends BaseStrategy> Map<Integer, AppliedChanges<T>> getAppliedChangesById(
            Map<ModelProperty<? super T, ?>, BiFunction<Object, Object, Boolean>> customModelEquals,
            Set<ModelProperty<?, ?>> sensitiveProperties,
            Map<Integer, ModelChanges<T>> validModelChanges,
            Map<Long, T> models) {
        var modelChangesWithChangedType = EntryStream.of(validModelChanges)
                .filterValues(mc -> isTypeChanged(models, mc))
                .toMap();

        var modelChangesWithUnchangedChangedType = EntryStream.of(validModelChanges)
                .removeKeys(modelChangesWithChangedType::containsKey)
                .toMap();

        Map<Integer, AppliedChanges<T>> appliedChangesWithChangedTypeByIndex =
                EntryStream.of(modelChangesWithChangedType)
                        .mapValues(mc -> getAppliedChangesWithChangedType(models, mc))
                        .toMap();

        var appliedChangesWithUnchangedTypeByIndex =
                getAppliedChangesForValidModelChanges(customModelEquals, sensitiveProperties,
                        modelChangesWithUnchangedChangedType, models);

        return EntryStream.of(appliedChangesWithChangedTypeByIndex)
                .append(appliedChangesWithUnchangedTypeByIndex)
                .toMap();
    }

    private <T extends BaseStrategy> AppliedChanges<T> getAppliedChangesWithChangedType(
            Map<Long, T> models, ModelChanges<T> mc) {
        var oldModel = models.get(mc.getId());
        StrategyName newType = mc.getChangedProp(BaseStrategy.TYPE);
        Set<ModelProperty<?, ?>> newModelProperties = PROPERTIES_BY_TYPE.get(newType);
        //noinspection unchecked
        T newModel = (T) TYPE_TO_STRATEGY_CLASS_SUPPLIER.get(newType).get();
        newModelProperties.forEach(
                property -> {
                    if (property.getModelClass().isAssignableFrom(oldModel.getClass())) {
                        property.copyRaw(oldModel, newModel);
                    }
                }
        );
        return mc.applyTo(newModel);
    }

    private <M extends BaseStrategy> Map<Integer, AppliedChanges<M>> getAppliedChangesForValidModelChanges(
            Map<ModelProperty<? super M, ?>, BiFunction<Object, Object, Boolean>> customModelEquals,
            Set<ModelProperty<?, ?>> sensitiveProperties,
            Map<Integer, ModelChanges<M>> validModelChanges,
            Map<Long, M> models) {
        return EntryStream.of(validModelChanges)
                .mapValues(oneModelChanges -> {
                    M model = models.get(oneModelChanges.getId());
                    var applicableSensitiveProperties = getApplicableSensitiveProperties(sensitiveProperties, model);
                    return oneModelChanges.applyTo(model, applicableSensitiveProperties, customModelEquals);
                })
                .toMap();
    }

    private <M extends BaseStrategy> Set<ModelProperty<? super M, ?>> getApplicableSensitiveProperties(
            Set<ModelProperty<?, ?>> sensitiveProperties,
            M model) {
        //noinspection unchecked
        return sensitiveProperties.stream()
                .filter(property -> property.getModelClass().isAssignableFrom(model.getClass()))
                .map(x -> (ModelProperty<? super M, ?>) x)
                .collect(Collectors.toSet());
    }
}
