package ru.yandex.direct.core.entity.strategy.type.common;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.Wallet;
import ru.yandex.direct.core.entity.campaign.service.WalletService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.strategy.container.StrategyAddOperationContainer;
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy;
import ru.yandex.direct.core.entity.strategy.service.CampaignUpdateHelper;
import ru.yandex.direct.core.entity.strategy.service.add.AbstractStrategyAddOperationTypeSupport;
import ru.yandex.direct.dbutil.model.ClientId;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.DEFAULT_ATTRIBUTION_MODEL;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Component
@ParametersAreNonnullByDefault
public class CommonStrategyAddTypeSupport extends AbstractStrategyAddOperationTypeSupport<CommonStrategy> {
    private final ClientService clientService;
    private final WalletService walletService;
    private final CampaignUpdateHelper campaignUpdateHelper;

    @Autowired
    public CommonStrategyAddTypeSupport(ClientService clientService,
                                        WalletService walletService,
                                        CampaignUpdateHelper campaignUpdateHelper) {
        this.clientService = clientService;
        this.walletService = walletService;
        this.campaignUpdateHelper = campaignUpdateHelper;
    }

    @Override
    public void onPreValidated(StrategyAddOperationContainer addContainer, List<CommonStrategy> models) {
        ClientId clientId = addContainer.getClientId();
        Client client = clientService.getClient(clientId);
        checkState(client != null, "Invalid clientId: " + clientId);

        List<Wallet> wallets = walletService.massGetWallets(List.of(clientId));
        checkState(wallets.size() == 1, "Client with id: " + clientId + " should have 1 wallet");
        Wallet wallet = wallets.get(0);

        models.forEach(strategy -> enrichStrategyOnPreValidated(addContainer, wallet, strategy));
    }

    private void enrichStrategyOnPreValidated(
            StrategyAddOperationContainer addContainer,
            Wallet wallet,
            CommonStrategy strategy) {
        strategy
                .withClientId(addContainer.getClientId().asLong())
                .withWalletId(wallet.getWalletCampaignId())
                .withStatusArchived(false)
                .withAttributionModel(nvl(strategy.getAttributionModel(), DEFAULT_ATTRIBUTION_MODEL))
                .withLastChange(addContainer.getNow());
    }


    @Override
    public void updateRelatedEntitiesInTransaction(DSLContext dslContext,
                                                   StrategyAddOperationContainer addModelContainer,
                                                   List<CommonStrategy> models) {

        campaignUpdateHelper.updateCampaignsInTransaction(dslContext, addModelContainer, models);
    }

    @Override
    public Class<CommonStrategy> getTypeClass() {
        return CommonStrategy.class;
    }
}
