package ru.yandex.direct.core.entity.strategy.type.common

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects
import ru.yandex.direct.core.entity.strategy.container.StrategyAddOperationContainer
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants.MAX_STRATEGIES_FOR_CLIENT_NUMBER
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants.MAX_UNARCHIVED_STRATEGIES_FOR_CLIENT_NUMBER
import ru.yandex.direct.core.entity.strategy.validation.add.AbstractStrategyAddValidationTypeSupport
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

@Component
class CommonStrategyAddValidationTypeSupport : AbstractStrategyAddValidationTypeSupport<CommonStrategy>() {
    override fun getTypeClass(): Class<CommonStrategy> = CommonStrategy::class.java

    override fun preValidate(
        container: StrategyAddOperationContainer,
        vr: ValidationResult<List<CommonStrategy>, Defect<*>>
    ): ValidationResult<List<CommonStrategy>, Defect<*>> {
        val validator = CommonStrategyValidatorProvider.createAddStrategyPreValidator(container)

        return ListValidationBuilder(vr)
            .checkEachBy(validator)
            .result
    }

    override fun validate(
        container: StrategyAddOperationContainer,
        vr: ValidationResult<List<CommonStrategy>, Defect<*>>
    ): ValidationResult<List<CommonStrategy>, Defect<*>> {
        updateClientStrategiesNumber(container, vr)

        val validator = CommonStrategyValidatorProvider.createAddStrategyValidator(container)

        val res = ListValidationBuilder(vr)
            .check(
                Constraint.fromPredicate(
                    {
                        container.clientUnarchivedStrategiesNumber <= MAX_UNARCHIVED_STRATEGIES_FOR_CLIENT_NUMBER
                    },
                    StrategyDefects.unarchivedStrategiesNumberLimitExceeded(MAX_UNARCHIVED_STRATEGIES_FOR_CLIENT_NUMBER)
                )
            )
            .check(
                Constraint.fromPredicate(
                    {
                        container.clientStrategiesNumber <= MAX_STRATEGIES_FOR_CLIENT_NUMBER
                    },
                    StrategyDefects.strategiesNumberLimitExceeded(MAX_STRATEGIES_FOR_CLIENT_NUMBER)
                )
            )
            .checkEachBy(validator)
            .result

        return res
    }

    private fun updateClientStrategiesNumber(
        container: StrategyAddOperationContainer,
        vr: ValidationResult<List<CommonStrategy>, Defect<*>>) {
        container.clientStrategiesNumber += vr.value.filter { it.isPublic == true }.size
        container.clientUnarchivedStrategiesNumber += vr.value.filter { it.isPublic == true && !it.statusArchived }.size
    }
}
