package ru.yandex.direct.core.entity.strategy.type.common

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel
import ru.yandex.direct.core.entity.campaign.model.CampaignWithAttributionModel
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDayBudget
import ru.yandex.direct.core.entity.campaign.model.CampaignWithEnableCpcHoldForbidden
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget
import ru.yandex.direct.core.entity.campaign.model.DayBudgetShowMode
import ru.yandex.direct.core.entity.campaign.model.DbStrategy
import ru.yandex.direct.core.entity.campaign.model.StrategyData
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstantsService
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.model.DefaultManualStrategy
import ru.yandex.direct.core.entity.strategy.model.StrategyAttributionModel
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAutobudget
import ru.yandex.direct.core.entity.strategy.model.StrategyWithDayBudget
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMeaningfulGoals
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMetrikaCounters
import ru.yandex.direct.core.entity.strategy.service.converter.StrategyToCampaignConverterSupport
import ru.yandex.direct.core.entity.strategy.service.converter.StrategyToCampaignConverterUtils
import ru.yandex.direct.model.ModelChanges
import java.math.BigDecimal
import java.time.LocalDateTime
import javax.annotation.ParametersAreNonnullByDefault

@Component
@ParametersAreNonnullByDefault
class CommonStrategyToCampaignConverterSupport(val campaignConstantsService: CampaignConstantsService) :
    StrategyToCampaignConverterSupport<CommonStrategy> {

    override fun fillCampaignTypeSpecificFields(
        strategy: CommonStrategy,
        now: LocalDateTime,
        campaign: CampaignWithPackageStrategy
    ) {
        if (strategy.id != campaign.strategyId) {
            campaign.strategyId = strategy.id
        }
        campaign.lastChange = now
        if (campaign is CampaignWithAttributionModel) {
            campaign.attributionModel = toCampaignStrategyAttributionModel(strategy.attributionModel)
        }

        fillDefaultValues(campaign)

        fill(
            StrategyToCampaignConverterUtils.createCampaignStrategyIfNull(campaign),
            StrategyToCampaignConverterUtils.createStrategyDataIfNull(campaign),
            strategy
        )
    }

    private fun fillDefaultValues(campaign: CampaignWithPackageStrategy) {
        if (campaign is CampaignWithDayBudget) {
            campaign.dayBudget = campaign.dayBudget ?: BigDecimal.ZERO
            campaign.dayBudgetDailyChangeCount = campaign.dayBudgetDailyChangeCount ?: 0
            campaign.dayBudgetShowMode = campaign.dayBudgetShowMode ?: DayBudgetShowMode.DEFAULT_
        }
    }

    override fun processCampaignModelChangesTypeSpecificProperties(
        strategy: CommonStrategy,
        now: LocalDateTime,
        modelChanges: ModelChanges<CampaignWithPackageStrategy>,
        campaignClass: Class<out CampaignWithPackageStrategy>
    ) {
        if (!modelChanges.isPropChanged(CampaignWithPackageStrategy.STRATEGY_ID)
            || modelChanges.getChangedProp(CampaignWithPackageStrategy.STRATEGY_ID) != strategy.id
        ) {
            modelChanges.process(strategy.id, CampaignWithPackageStrategy.STRATEGY_ID)
        }

        modelChanges.process(now, CampaignWithPackageStrategy.LAST_CHANGE)
        if (CampaignWithAttributionModel::class.java.isAssignableFrom(campaignClass)) {
            modelChanges.castModel(CampaignWithAttributionModel::class.java).process(
                toCampaignStrategyAttributionModel(strategy.attributionModel),
                CampaignWithAttributionModel.ATTRIBUTION_MODEL
            )
        }
        dropFieldsIfNeed(strategy::class.java, modelChanges, campaignClass)

        fill(
            StrategyToCampaignConverterUtils.createCampaignStrategyIfNull(modelChanges),
            StrategyToCampaignConverterUtils.createStrategyDataIfNull(modelChanges),
            strategy
        )
    }

    override fun getTypeClass() = CommonStrategy::class.java

    private fun dropFieldsIfNeed(
        strategyClass: Class<out CommonStrategy>,
        modelChanges: ModelChanges<CampaignWithPackageStrategy>,
        campaignClass: Class<*>
    ) {
        if (!StrategyWithDayBudget::class.java.isAssignableFrom(strategyClass) &&
            CampaignWithDayBudget::class.java.isAssignableFrom(campaignClass)
        ) {
            val castedChanges = modelChanges.castModel(CampaignWithDayBudget::class.java)
            castedChanges.process(BigDecimal.ZERO, CampaignWithDayBudget.DAY_BUDGET)
            castedChanges.process(
                DayBudgetShowMode.DEFAULT_,
                CampaignWithDayBudget.DAY_BUDGET_SHOW_MODE
            )
        }

        if (!StrategyWithMeaningfulGoals::class.java.isAssignableFrom(strategyClass) &&
            CampaignWithMeaningfulGoals::class.java.isAssignableFrom(campaignClass)
        ) {
            val castedChanges = modelChanges.castModel(CampaignWithMeaningfulGoals::class.java)
            castedChanges.process(null, CampaignWithMeaningfulGoals.MEANINGFUL_GOALS)
        }

        if (!DefaultManualStrategy::class.java.isAssignableFrom(strategyClass) &&
            !CampaignWithEnableCpcHoldForbidden::class.java.isAssignableFrom(campaignClass)
        ) {
            modelChanges.process(false, CampaignWithPackageStrategy.ENABLE_CPC_HOLD)
        }

        if (!StrategyWithMetrikaCounters::class.java.isAssignableFrom(strategyClass) &&
            CampaignWithMetrikaCounters::class.java.isAssignableFrom(campaignClass)
        ) {
            val castedModelChanges = modelChanges.castModel(CampaignWithMetrikaCounters::class.java)
            castedModelChanges.process(null, CampaignWithMetrikaCounters.METRIKA_COUNTERS)
        }
    }

    private fun fill(campaignStrategy: DbStrategy, strategyData: StrategyData, strategy: CommonStrategy) {
        campaignStrategy.strategyName = toCampaignStrategyName(strategy.type)
        campaignStrategy.autobudget =
            if (strategy is StrategyWithAutobudget) CampaignsAutobudget.YES else CampaignsAutobudget.NO

        strategyData.name = StrategyName.toSource(campaignStrategy.strategyName)?.literal
        strategyData.version = 1
    }

    companion object {
        fun toCampaignStrategyName(type: ru.yandex.direct.core.entity.strategy.model.StrategyName): StrategyName {
            return StrategyName.valueOf(type.name)
        }

        private fun toCampaignStrategyAttributionModel(attributionModel: StrategyAttributionModel): CampaignAttributionModel {
            return CampaignAttributionModel.valueOf(attributionModel.name)
        }
    }
}
