package ru.yandex.direct.core.entity.strategy.type.common

import java.time.LocalDateTime
import java.util.function.Supplier
import org.jooq.DSLContext
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.strategy.container.AbstractStrategyOperationContainer
import ru.yandex.direct.core.entity.strategy.container.StrategyRepositoryContainer
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.repository.StrategyModifyRepository
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges

@Service
class CommonStrategyUpdateLastChangeService(
    private val strategyTypedRepository: StrategyTypedRepository,
    private val strategyModifyRepository: StrategyModifyRepository,
    private val featureService: FeatureService
) {

    companion object {
        fun getOldStrategyIdsFromCampaignChanges(campaignsChanges: List<AppliedChanges<CampaignWithPackageStrategy>>) =
            campaignsChanges
                .filter(this::isStrategyIdChanged)
                .flatMap { listOf(it.getOldValue(CampaignWithPackageStrategy.STRATEGY_ID), it.getNewValue(CampaignWithPackageStrategy.STRATEGY_ID)) }
                .filterNotNull()

        private fun isStrategyIdChanged(campaignChanges: AppliedChanges<CampaignWithPackageStrategy>) = 
            campaignChanges.model.strategyId != null && campaignChanges.model.strategyId != 0L
                    && campaignChanges.changed(CampaignWithPackageStrategy.STRATEGY_ID)

        private fun getOldStrategyIdsFromCampaigns(campaigns: Collection<CampaignWithPackageStrategy>) =
            campaigns.toList()
                .map { it.strategyId }
                .filter { it != null && it != 0L }

        private fun toStrategyWithNewLastChange(strategy: CommonStrategy, currentTime: LocalDateTime) =
            ModelChanges.build(strategy.id, CommonStrategy::class.java, CommonStrategy.LAST_CHANGE, currentTime)
                .applyTo(strategy)
    }

    fun updateLastChangeForCampaignsStrategies(
        dslContext: DSLContext,
        container: AbstractStrategyOperationContainer,
        relatedCampaigns: Collection<CampaignWithPackageStrategy>
    ) {
        updateLastChangeForCampaignsStrategies(dslContext, container) {
            getOldStrategyIdsFromCampaigns(relatedCampaigns)
        }
    }

    fun updateLastChangeForCampaignsStrategies(
        dslContext: DSLContext,
        container: AbstractStrategyOperationContainer,
        campaignsAppliedChanges: List<AppliedChanges<CampaignWithPackageStrategy>>
    ) {
        updateLastChangeForCampaignsStrategies(dslContext, container) {
            getOldStrategyIdsFromCampaignChanges(campaignsAppliedChanges)
        }
    }

    private fun updateLastChangeForCampaignsStrategies(
        dslContext: DSLContext,
        operationContainer: AbstractStrategyOperationContainer,
        strategyIdsSupplier: Supplier<List<Long>>
    ) {
        val isStageTwoEnabled = featureService.isEnabledForClientId(
            operationContainer.clientId, FeatureName.PACKAGE_STRATEGIES_STAGE_TWO
        )

        if (!isStageTwoEnabled) return

        val previousStrategyIds = strategyIdsSupplier.get()

        if (previousStrategyIds.isEmpty()) return

        val currentTime = LocalDateTime.now()

        val models = this.strategyTypedRepository.getTyped(dslContext, previousStrategyIds)

        val changes = models
            .filterIsInstance(CommonStrategy::class.java)
            .map { toStrategyWithNewLastChange(it, currentTime) }

        val container = StrategyRepositoryContainer(
            operationContainer.shard, operationContainer.clientId, mapOf(), true
        )
        this.strategyModifyRepository.update(dslContext, container, changes);
    }
}