package ru.yandex.direct.core.entity.strategy.type.common

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects
import ru.yandex.direct.core.entity.strategy.container.StrategyUpdateOperationContainer
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants
import ru.yandex.direct.core.entity.strategy.validation.update.AbstractStrategyUpdateValidationTypeSupport
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints.unique
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

@Component
class CommonStrategyUpdateValidationTypeSupport : AbstractStrategyUpdateValidationTypeSupport<CommonStrategy>() {
    override fun getTypeClass(): Class<CommonStrategy> = CommonStrategy::class.java

    override fun preValidate(
        container: StrategyUpdateOperationContainer,
        vr: ValidationResult<List<ModelChanges<CommonStrategy>>, Defect<*>>
    ): ValidationResult<List<ModelChanges<CommonStrategy>>, Defect<*>> {
        val validator = CommonStrategyValidatorProvider.createUpdateStrategyPreValidator(container)

        return ListValidationBuilder(vr)
            .checkEach(unique { it: ModelChanges<CommonStrategy> -> it.id })
            .checkEachBy(validator)
            .result
    }

    override fun validateBeforeApply(
        container: StrategyUpdateOperationContainer,
        vr: ValidationResult<List<ModelChanges<CommonStrategy>>, Defect<*>>,
        unmodifiedModels: Map<Long, CommonStrategy>
    ): ValidationResult<List<ModelChanges<CommonStrategy>>, Defect<*>> {
        updateClientStrategiesNumber(container, vr, unmodifiedModels)

        val commonStrategyContainer = CommonStrategyUpdateOperationContainer(unmodifiedModels)
        val validator =
            CommonStrategyValidatorProvider.createUpdateStrategyBeforeApplyValidator(container, commonStrategyContainer)

        return ListValidationBuilder(vr)
            .check(
                Constraint.fromPredicate(
                    {
                        container.clientUnarchivedStrategiesNumber <= StrategyConstants.MAX_UNARCHIVED_STRATEGIES_FOR_CLIENT_NUMBER
                    },
                    StrategyDefects.unarchivedStrategiesNumberLimitExceeded(StrategyConstants.MAX_UNARCHIVED_STRATEGIES_FOR_CLIENT_NUMBER)
                )
            )
            .check(
                Constraint.fromPredicate(
                    {
                        container.clientStrategiesNumber <= StrategyConstants.MAX_STRATEGIES_FOR_CLIENT_NUMBER
                    },
                    StrategyDefects.strategiesNumberLimitExceeded(StrategyConstants.MAX_STRATEGIES_FOR_CLIENT_NUMBER)
                )
            )
            .checkEachBy(validator)
            .result
    }

    override fun validate(
        container: StrategyUpdateOperationContainer,
        vr: ValidationResult<List<CommonStrategy>, Defect<*>>
    ): ValidationResult<List<CommonStrategy>, Defect<*>> {

        val validator = CommonStrategyValidatorProvider.createUpdateStrategyValidator(container)

        return ListValidationBuilder(vr)
            .checkEachBy(validator)
            .result
    }

    private fun updateClientStrategiesNumber(
        container: StrategyUpdateOperationContainer,
        vr: ValidationResult<List<ModelChanges<CommonStrategy>>, Defect<*>>,
        unmodifiedModels: Map<Long, CommonStrategy>) {
        container.clientStrategiesNumber += vr.value.filter {
            (it.isPropChanged(CommonStrategy.IS_PUBLIC)
                && it.getChangedProp(CommonStrategy.IS_PUBLIC) == true
                && unmodifiedModels[it.id]?.isPublic == false)
        }.size

        container.clientUnarchivedStrategiesNumber += vr.value.filter {
            (it.isPropChanged(CommonStrategy.IS_PUBLIC)
                && it.getChangedProp(CommonStrategy.IS_PUBLIC) == true
                && unmodifiedModels[it.id]?.isPublic == false
                && unmodifiedModels[it.id]?.statusArchived == false)
        }.size
    }
}
