package ru.yandex.direct.core.entity.strategy.type.common

import ru.yandex.direct.core.entity.campaign.service.validation.CampaignValidators.fieldCantBeModified
import ru.yandex.direct.core.entity.strategy.container.StrategyAddOperationContainer
import ru.yandex.direct.core.entity.strategy.container.StrategyUpdateOperationContainer
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.validation.AbstractStrategyValidatorProviderWithTypeSpecificContainer
import ru.yandex.direct.core.validation.defects.RightsDefects.forbiddenToChange
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder

object CommonStrategyValidatorProvider :
    AbstractStrategyValidatorProviderWithTypeSpecificContainer<CommonStrategy, CommonStrategyUpdateOperationContainer>() {
    override fun createAddStrategyPreValidator(container: StrategyAddOperationContainer): Validator<CommonStrategy, Defect<*>> =
        Validator { commonStrategy: CommonStrategy ->
            val vb = ModelItemValidationBuilder.of(commonStrategy)

            vb.item(CommonStrategy.CLIENT_ID)
                .checkBy(ClientIdValidator(container.options.isCampaignToPackageStrategyOneshot))
            vb.item(CommonStrategy.IS_PUBLIC)
                .checkBy(IsPublicPreValidator(container.wallet != null))
            vb.item(CommonStrategy.WALLET_ID)
                .checkBy(WalletIdValidator(container.options.isCampaignToPackageStrategyOneshot))
            vb.item(CommonStrategy.CIDS)
                .checkBy(LinkedCampaignIdsPreValidator(container.campaigns(commonStrategy)))
            vb.item(CommonStrategy.CIDS)
                .checkBy(
                    LinkedCampaignIdsValidator(
                        container.campaigns(commonStrategy),
                        commonStrategy.isPublic,
                        commonStrategy.type
                    )
                )

            vb.checkBy(LinkedCampaignTypesValidator(container.campaigns(commonStrategy)))

            vb.result
        }

    override fun createAddStrategyValidator(container: StrategyAddOperationContainer): Validator<CommonStrategy, Defect<*>> =
        Validator { commonStrategy: CommonStrategy ->
            val vb = ModelItemValidationBuilder.of(commonStrategy)

            vb.item(CommonStrategy.TYPE).checkBy(TypeValidator(commonStrategy))
            vb.item(CommonStrategy.NAME).checkBy(NameValidator)
            vb.item(CommonStrategy.ATTRIBUTION_MODEL).checkBy(AttributionModelValidator)
            vb.item(CommonStrategy.IS_PUBLIC)
                .checkBy(
                    IsPublicValidator(
                        campaigns = container.campaigns(commonStrategy).map { it.id })
                )
            vb.item(CommonStrategy.CIDS)
                .checkBy(NumberOfCidsValidator(container.options.maxNumberOfCids))

            vb.checkBy(PublicStrategyTypeValidator(), When.isTrue(commonStrategy.isPublic == true))
                .checkBy(StrategyToStrategyTypeValidator())

            vb.result
        }

    override fun createUpdateStrategyPreValidator(container: StrategyUpdateOperationContainer): Validator<ModelChanges<CommonStrategy>, Defect<*>> =
        Validator { mc: ModelChanges<CommonStrategy> ->
            val vb = ModelChangesValidationBuilder.of(mc)
            vb.checkBy(fieldCantBeModified(CommonStrategy.CLIENT_ID, forbiddenToChange()))
            vb.checkBy(fieldCantBeModified(CommonStrategy.WALLET_ID, forbiddenToChange()))
            // запрещаем линковать uac кампании к публичным стратегиям
            vb.item(CommonStrategy.CIDS)
                .checkBy(
                    LinkedCampaignIdsPreValidator(container.typedCampaignsMap.getOrDefault(mc.id, listOf())),
                    When.isTrue(mc.isPropChanged(CommonStrategy.CIDS))
                )
            vb.item(CommonStrategy.IS_PUBLIC)
                .checkBy(
                    IsPublicPreValidator(container.wallet != null),
                    When.isTrue(mc.isPropChanged(CommonStrategy.IS_PUBLIC))
                )
            vb.result
        }

    override fun createUpdateStrategyBeforeApplyValidator(
        container: StrategyUpdateOperationContainer,
        typeSpecificContainer: CommonStrategyUpdateOperationContainer
    ): Validator<ModelChanges<CommonStrategy>, Defect<*>> =
        Validator { mc: ModelChanges<CommonStrategy> ->
            val vb = ModelChangesValidationBuilder.of(mc)

            val unmodifiedModel = typeSpecificContainer.unmodifiedModels[mc.id]

            if (unmodifiedModel != null) {
                vb.item(CommonStrategy.IS_PUBLIC)
                    .checkBy(IsPublicValidator(container.campaignsById(mc.id)?.map { it.id }, unmodifiedModel.isPublic))
                vb.item(CommonStrategy.CIDS)
                    .checkBy(LinkedCampaignIdsBeforeApplyValidator(unmodifiedModel.cids?: emptyList()))
            }

            vb.result
        }

    override fun createUpdateStrategyValidator(container: StrategyUpdateOperationContainer): Validator<CommonStrategy, Defect<*>> =
        Validator { commonStrategy: CommonStrategy ->
            val vb = ModelItemValidationBuilder.of(commonStrategy)

            vb.item(CommonStrategy.TYPE).checkBy(TypeValidator(commonStrategy))
            vb.item(CommonStrategy.NAME).checkBy(NameValidator)
            vb.item(CommonStrategy.ATTRIBUTION_MODEL).checkBy(AttributionModelValidator)
            vb.item(CommonStrategy.STATUS_ARCHIVED).checkBy(StatusArchivedUpdateValidator)
            vb.item(CommonStrategy.CIDS)
                .checkBy(
                    LinkedCampaignIdsValidator(
                        container.campaigns(commonStrategy),
                        commonStrategy.isPublic,
                        commonStrategy.type
                    )
                )
            vb.item(CommonStrategy.CIDS)
                .checkBy(NumberOfCidsValidator(container.options.maxNumberOfCids))

            vb.checkBy(LinkedCampaignTypesValidator(container.campaigns(commonStrategy)))
                .checkBy(PublicStrategyTypeValidator(), When.isTrue(commonStrategy.isPublic == true))
                .checkBy(StrategyToStrategyTypeValidator())

            vb.result
        }
}
