package ru.yandex.direct.core.entity.strategy.type.common

import ru.yandex.direct.core.entity.campaign.model.CampaignSource
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.inconsistentStrategyToCampaignType
import ru.yandex.direct.core.entity.strategy.model.StrategyName
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants.AVAILABLE_CAMPAIGN_TYPES_FOR_PUBLIC_STRATEGY
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants.CAMPAIGN_TYPES_AVAILABLE_FOR_STRATEGY_TYPES
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize
import ru.yandex.direct.validation.defect.CommonDefects.invalidValue
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

class LinkedCampaignIdsValidator(
    private val currentCampaigns: List<CommonCampaign>,
    private val isPublic: Boolean?,
    private val strategyType: StrategyName
) :
    Validator<List<Long>?, Defect<*>> {
    override fun apply(t: List<Long>?): ValidationResult<List<Long>?, Defect<*>> {
        val vb = ListValidationBuilder.of<Long, Defect<*>>(t)
        val campaignSourceById = currentCampaigns.associate { it.id to it.source }
            .toMap()
        val campaignTypeById = currentCampaigns.associate { it.id to it.type }
            .toMap()

        vb
            .check(maxListSize(1), When.isTrue(true != isPublic))
            .checkEach(
                Constraint.fromPredicate({ campaignSourceById[it] != CampaignSource.UAC }, invalidValue()),
                When.isTrue(true == isPublic)
            )
            .checkEach(campaignTypeSupported(campaignTypeById), When.isTrue(true == isPublic))
            .checkEach(campaignTypeAvailableByStrategy(campaignTypeById))

        return vb.result
    }

    private fun campaignTypeSupported(campaignTypeById: Map<Long, CampaignType>): Constraint<Long, Defect<*>> =
        Constraint.fromPredicate(
            {
                !campaignTypeById.containsKey(it) || AVAILABLE_CAMPAIGN_TYPES_FOR_PUBLIC_STRATEGY.contains(campaignTypeById[it])
            },
            inconsistentStrategyToCampaignType()
        )

    private fun campaignTypeAvailableByStrategy(campaignTypeById: Map<Long, CampaignType>): Constraint<Long, Defect<*>> =
        Constraint.fromPredicate(
            {
                CAMPAIGN_TYPES_AVAILABLE_FOR_STRATEGY_TYPES[strategyType]?.contains(campaignTypeById[it]) ?: false
            },
            inconsistentStrategyToCampaignType()
        )

}
