package ru.yandex.direct.core.entity.strategy.type.withbid

import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.weekBudgetLessThan
import ru.yandex.direct.core.entity.strategy.model.StrategyName
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.NumberConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder
import java.math.BigDecimal

class BidValidator(
    currency: Currency,
    private val sum: BigDecimal?,
    campaignsType: CampaignType?,
    private val strategyType: StrategyName,
    private val isPayForConversionEnabled: Boolean
) : Validator<BigDecimal, Defect<*>> {

    private val minValue = when (campaignsType) {
        CampaignType.PERFORMANCE -> currency.minCpcCpaPerformance
        else -> currency.minPrice
    }

    private val maxValue = when (strategyType) {
        StrategyName.AUTOBUDGET_WEEK_BUNDLE -> currency.maxPrice
        else -> currency.maxAutobudgetBid
    }

    override fun apply(value: BigDecimal?): ValidationResult<BigDecimal, Defect<*>> {
        return ModelItemValidationBuilder.of<BigDecimal, Defect<*>>(value)
            .check(CommonConstraints.isNull(), When.isTrue(isBidNotAllowable()))
            .check(CommonConstraints.notNull(), When.isFalse(isBidNotAllowable()))
            .check(NumberConstraints.notLessThan(minValue))
            .check(NumberConstraints.notGreaterThan(maxValue))
            .check(NumberConstraints.notGreaterThan(sum), weekBudgetLessThan(), When.isTrue(sum != null))
            .result
    }

    private fun isBidNotAllowable(): Boolean =
        isPayForConversionEnabled && STRATEGY_TYPES_BID_NOT_ALLOWABLE.contains(strategyType)

    companion object {
        private val STRATEGY_TYPES_BID_NOT_ALLOWABLE = setOf(
            StrategyName.AUTOBUDGET_AVG_CPI,
            StrategyName.AUTOBUDGET_AVG_CPA,
            StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP
        )
    }
}
