package ru.yandex.direct.core.entity.strategy.type.withcampaignids;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.TableField;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.strategy.model.StrategyWithCampaignIds;
import ru.yandex.direct.core.entity.strategy.repository.AbstractStrategyRepositoryTypeSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;

import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Strategies.STRATEGIES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class StrategyWithCampaignIdsRepositoryTypeSupport extends AbstractStrategyRepositoryTypeSupport<StrategyWithCampaignIds> {

    private final JooqMapper<StrategyWithCampaignIds> mapper = createMapper();

    protected StrategyWithCampaignIdsRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public JooqMapper<StrategyWithCampaignIds> createMapper() {
        return JooqMapperBuilder.<StrategyWithCampaignIds>builder()
                .map(property(StrategyWithCampaignIds.ID, STRATEGIES.STRATEGY_ID))
                .build();
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<StrategyWithCampaignIds> strategies) {
        Set<Long> strategyIds = listToSet(strategies, StrategyWithCampaignIds::getId);
        var strategyIdToCids = getStrategyIdToCids(dslContext, strategyIds);

        strategies.forEach(strategy -> strategy.setCids(strategyIdToCids.get(strategy.getId())));
    }

    private Map<Long, List<Long>> getStrategyIdToCids(DSLContext dslContext, Set<Long> strategyIds) {
        return dslContext.select(CAMPAIGNS.STRATEGY_ID, CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.STRATEGY_ID.in(strategyIds))
                .fetchGroups(CAMPAIGNS.STRATEGY_ID, CAMPAIGNS.CID);
    }

    @Override
    public Set<TableField<?, Long>> getPrimaryKeys() {
        return Set.of(STRATEGIES.STRATEGY_ID);
    }

    @Override
    public Collection<Field<?>> getFields() {
        return mapper.getFieldsToRead();
    }

    @Override
    public <M extends StrategyWithCampaignIds> void fillFromRecord(M model, Record record) {
        mapper.fromDb(record, model);
    }

    @Override
    public Class<StrategyWithCampaignIds> getTypeClass() {
        return StrategyWithCampaignIds.class;
    }
}
