package ru.yandex.direct.core.entity.strategy.type.withconversion

import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BY_ALL_GOALS_GOAL_ID
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.allGoalsOptimizationProhibited
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.strategy.model.StrategyName
import ru.yandex.direct.core.entity.strategy.validation.StrategyValidationUtils.campaignTypeNullOr
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.CommonConstraints.validId
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.validateObject

class GoalIdValidator(private val container: ValidationContainer) : Validator<Long?, Defect<*>> {

    override fun apply(goalId: Long?) =
        validateObject(goalId) {
            checkBy(this@GoalIdValidator::validateGoalIdIsAllowedForSocialClient)
            when {
                container.strategyType == StrategyName.AUTOBUDGET_AVG_CPI ->
                    checkBy(this@GoalIdValidator::validateMobileContentGoalId, When.notNull())

                container.campaignType == CampaignType.MOBILE_CONTENT ->
                    checkBy(this@GoalIdValidator::validateMobileContentGoalId)

                goalId == BY_ALL_GOALS_GOAL_ID ->
                    checkByFunction { allGoalsOptimizationProhibited() }

                //Do nothing
                goalId == MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID -> {
                }

                container.strategyType == StrategyName.AUTOBUDGET -> {
                    check(notNull(), When.isTrue(container.campaignType == CampaignType.PERFORMANCE))
                    checkBy(this@GoalIdValidator::validateMetrikaGoalId, When.notNull())
                }

                else ->
                    checkBy(this@GoalIdValidator::validateGoalId)
            }
        }

    private fun validateGoalId(goalId: Long?): ValidationResult<Long?, Defect<*>> =
        when (container.campaignType) {
            null -> {
                // Если не задан тип кампании, то это может быть РМП и соответственно мобильные цели допустимы.
                validateObject(goalId) {
                    check(notNull())
                    check(validId())
                    check(
                        fromPredicate(container.goalIds::contains, objectNotFound()),
                        When.isValidAnd(When.isFalse(container.isCopy || container.allGoalsAreAvailable))
                    )
                }
            }
            CampaignType.MOBILE_CONTENT -> {
                validateMobileContentGoalId(goalId)
            }
            else -> {
                validateMetrikaGoalId(goalId)
            }
        }

    private fun validateMetrikaGoalId(goalId: Long?) =
        validateObject(goalId) {
            check(notNull())
            check(validId())
            check(
                fromPredicate(container.availableMetrikaGoalIds::contains, objectNotFound()),
                When.isValidAnd(When.isFalse(container.isCopy || container.allGoalsAreAvailable))
            )
        }

    private fun validateMobileContentGoalId(goalId: Long?) =
        validateObject(goalId) {
            check(notNull(), When.isTrue(container.strategyType != StrategyName.AUTOBUDGET))
            check(validId())
            check(
                fromPredicate(container.mobileGoalIds::contains, objectNotFound()),
                When.isFalse(container.isCampaignMigration)
            )
        }

    private fun validateGoalIdIsAllowedForSocialClient(goalId: Long?) =
        validateObject(goalId) {
            check(
                fromPredicate({ !isSocialClient() }, StrategyDefects.inconsistentStrategyToCampaignType()),
                When.notNullAnd(When.isTrue(!container.isCampaignMigration && container.strategyType == StrategyName.AUTOBUDGET))
            )
        }

    private fun isSocialClient(): Boolean {
        return container.availableFeatures.contains(FeatureName.SOCIAL_ADVERTISING)
    }

    companion object {
        private val CAMPAIGN_TYPES_SUPPORTS_METRIKA_GOALS = setOf(
            CampaignType.TEXT,
            CampaignType.CONTENT_PROMOTION,
            CampaignType.CPM_BANNER,
            CampaignType.CPM_PRICE,
            CampaignType.CPM_YNDX_FRONTPAGE,
            CampaignType.DYNAMIC,
            CampaignType.INTERNAL_AUTOBUDGET,
            CampaignType.INTERNAL_DISTRIB,
            CampaignType.INTERNAL_FREE,
            CampaignType.MCBANNER,
            CampaignType.PERFORMANCE
        )

        data class ValidationContainer(
            val goals: Set<Goal>,
            val campaignType: CampaignType?,
            val strategyType: StrategyName,
            val availableFeatures: Set<FeatureName>,
            val requestFromInternalNetwork: Boolean,
            val isCopy: Boolean = false,
            val isCampaignMigration: Boolean = false
        ) {

            val availableMetrikaGoalIds = if (campaignTypeNullOr(campaignType, CAMPAIGN_TYPES_SUPPORTS_METRIKA_GOALS)) {
                goals.filterNot(Goal::getIsMobileGoal).map(Goal::getId).toSet()
            } else {
                emptySet()
            }

            val allGoalsAreAvailable =
                isCampaignMigration || (requestFromInternalNetwork && !availableFeatures.contains(FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED))

            val mobileGoalIds = goals
                .filter(Goal::getIsMobileGoal)
                .map(Goal::getId)
                .toSet()

            val goalIds = goals.map(Goal::getId).toSet()
        }
    }
}
