package ru.yandex.direct.core.entity.strategy.type.withdaybudget

import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.dayBudgetShowModeOverridenByWallet
import ru.yandex.direct.core.entity.strategy.model.StrategyDayBudgetShowMode
import ru.yandex.direct.core.entity.strategy.validation.StrategyValidationUtils.isWalletHasDayBudget
import ru.yandex.direct.dbschema.ppc.enums.CampaignsDayBudgetShowMode
import ru.yandex.direct.utils.NumberUtils.greaterThanZero
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.validateObject
import java.math.BigDecimal

class DayBudgetShowModeValidator(private val container: ValidationContainer) : Validator<StrategyDayBudgetShowMode?, Defect<*>> {
    override fun apply(t: StrategyDayBudgetShowMode?): ValidationResult<StrategyDayBudgetShowMode?, Defect<*>> =
        validateObject(t) {
            check(notNull(), When.isTrue(container.isCampaignTypeSupports))
            checkBy(this@DayBudgetShowModeValidator::validate, When.isTrue(t != null))
        }

    private fun validate(t: StrategyDayBudgetShowMode?): ValidationResult<StrategyDayBudgetShowMode?, Defect<*>> =
        validateObject(t) {
            weakCheck(fromPredicate({ it == container.walletDayBudgetShowMode }, dayBudgetShowModeOverridenByWallet()),
                When.isValidAnd(When.isTrue(greaterThanZero(container.dayBudget) && isWalletHasDayBudget(container.wallet))))
        }

    companion object {
        data class ValidationContainer(val wallet: Campaign?,
                                       val dayBudget: BigDecimal?,
                                       val campaignType: CampaignType?) {
            val isCampaignTypeSupports = SUPPORTED_FOR_CAMPAIGN_TYPES.contains(campaignType)

            val walletDayBudgetShowMode = wallet?.dayBudgetShowMode?.let {
                when (it) {
                    CampaignsDayBudgetShowMode.default_ -> StrategyDayBudgetShowMode.DEFAULT_
                    CampaignsDayBudgetShowMode.stretched -> StrategyDayBudgetShowMode.STRETCHED
                }
            }
        }

        private val SUPPORTED_FOR_CAMPAIGN_TYPES = setOf(
            CampaignType.CONTENT_PROMOTION,
            CampaignType.CPM_BANNER,
            CampaignType.DYNAMIC,
            CampaignType.MCBANNER,
            CampaignType.MOBILE_CONTENT,
            CampaignType.TEXT
        )
    }
}
