package ru.yandex.direct.core.entity.strategy.type.withdaybudget

import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDayBudget
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.dayBudgetCanBeSet
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.dayBudgetNotSupportedWithStrategy
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.dayBudgetOverridenByWallet
import ru.yandex.direct.core.entity.strategy.validation.StrategyValidationUtils.campaignTypeNullOr
import ru.yandex.direct.core.entity.strategy.validation.StrategyValidationUtils.isWalletHasDayBudget
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.utils.NumberUtils
import ru.yandex.direct.utils.NumberUtils.isZero
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.NumberConstraints.inRange
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.validateObject
import java.math.BigDecimal

class DayBudgetValidator(private val container: ValidationContainer) : Validator<BigDecimal?, Defect<*>> {
    override fun apply(t: BigDecimal?): ValidationResult<BigDecimal?, Defect<*>> =
        validateObject(t) {
            check(fromPredicate({ isZero(t) }, dayBudgetNotSupportedWithStrategy()), When.isFalse(campaignTypeNullOr(container.campaignType, SUPPORTED_FOR_CAMPAIGN_TYPES::contains)))
            checkBy(this@DayBudgetValidator::validate, When.isFalse(t == null || isZero(t)))
        }

    private fun validate(t: BigDecimal?): ValidationResult<BigDecimal?, Defect<*>> =
        validateObject(t) {
            check(notNull())
            check(dayBudgetCanBeSet(container.maxDailyBudgetChangeCount), When.isValid())
            check(inRange(container.currency.minDayBudget, container.currency.maxDailyBudgetAmount), When.isValidAnd(When.isTrue(
                NumberUtils.greaterThanZero(t))))
            weakCheck(fromPredicate({ container.isDayBudgetOverridenByWallet(it) }, dayBudgetOverridenByWallet()),
                When.isValidAnd(When.isTrue(NumberUtils.greaterThanZero(t)
                    && isWalletHasDayBudget(container.wallet))))
        }

    companion object {
        data class ValidationContainer(val campaignType: CampaignType?,
                                       val currency: Currency,
                                       val strategyDailyBudgetChangeCount: Int?,
                                       val linkedCampaigns: List<CampaignWithDayBudget>,
                                       val wallet: Campaign?) {

            val maxDailyBudgetChangeCount: Int =
                (linkedCampaigns.mapNotNull { it.dayBudgetDailyChangeCount } + (strategyDailyBudgetChangeCount ?: 0)).maxOrNull() ?: 0

            fun isDayBudgetOverridenByWallet(dayBudget: BigDecimal?): Boolean =
                dayBudget == null || wallet?.dayBudget == null ||
                    isZero(dayBudget) || wallet.dayBudget >= dayBudget

        }

        private val SUPPORTED_FOR_CAMPAIGN_TYPES = setOf(
            CampaignType.CONTENT_PROMOTION,
            CampaignType.CPM_BANNER,
            CampaignType.CPM_YNDX_FRONTPAGE,
            CampaignType.DYNAMIC,
            CampaignType.MCBANNER,
            CampaignType.MOBILE_CONTENT,
            CampaignType.PERFORMANCE,
            CampaignType.TEXT
        )
    }
}
