package ru.yandex.direct.core.entity.strategy.type.withmeaningfulgoals

import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ENGAGED_SESSION_GOAL_ID
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_MAX_COUNT
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.incorrectSetOfMobileGoals
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.unableToUseCurrentMeaningfulGoalsForOptimization
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.strategy.model.StrategyName
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.Constraint.fromPredicateOfNullable
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize
import ru.yandex.direct.validation.constraint.CollectionConstraints.unique
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import java.util.function.Function

class MeaningfulGoalsValidator(private val container: ValidationContainer) :
    Validator<List<MeaningfulGoal>, Defect<*>> {

    override fun apply(t: List<MeaningfulGoal>?): ValidationResult<List<MeaningfulGoal>, Defect<*>> {
        val vb = ListValidationBuilder.of<MeaningfulGoal, Defect<*>>(t)

        if (container.currency == null) {
            return vb.result
        }
        val meaningfulGoalValidator = MeaningfulGoalValidator(
            MeaningfulGoalValidator.Companion.ValidationContainer(
                container.currency,
                container.goals.map(Goal::getId).toSet() + ENGAGED_SESSION_GOAL_ID,
                container.allGoalsAreAvailable,
                container.allowMeaningfulGoalValueFromMetrika,
                container.strategyType,
                container.isCopy
            )
        )
        vb.check(maxListSize(MEANINGFUL_GOALS_MAX_COUNT), When.notNull())
            .checkEach(unique(MeaningfulGoal::getGoalId), When.notNull())
            .checkEachBy(meaningfulGoalValidator::apply, When.notNull())

        vb.check(
            fromPredicateOfNullable(
                this::notOnlyEngagedSessionGoalId,
                unableToUseCurrentMeaningfulGoalsForOptimization()
            ),
            When.isTrue(container.isMeaningfulGoalsCheckRequired())
        )

        vb.check(fromPredicate(this::checkMobileGoals, incorrectSetOfMobileGoals()))

        return vb.result
    }

    private fun checkMobileGoals(meaningfulGoals: List<MeaningfulGoal>): Boolean {
        val goalIdToMap = container.goals.associateBy(Goal::getId)
        val mobileAppIds = meaningfulGoals
            .mapNotNull { mg -> goalIdToMap[mg.goalId] }
            .filter { it.isMobileGoal == true }
            .mapNotNull(Goal::getMobileAppId)

        return if (mobileAppIds.isEmpty()) {
            true
        } else {
            container.mobileContentProvider.apply(mobileAppIds.toSet())
                .groupBy(MobileContent::getOsType)
                .filterValues { it.size > 1 }
                .isEmpty()
        }
    }

    private fun notOnlyEngagedSessionGoalId(meaningfulGoals: List<MeaningfulGoal>?) =
        meaningfulGoals != null && meaningfulGoals.find { it.goalId != ENGAGED_SESSION_GOAL_ID } != null

    companion object {
        data class ValidationContainer(
            val currency: Currency?,
            val goals: Set<Goal>,
            val availableFeatures: Set<FeatureName>,
            val strategyType: StrategyName,
            var mobileContentProvider: Function<Set<Long>, List<MobileContent>>,
            val isCopy: Boolean,
            val isRequestFromInternalNetwork: Boolean,
            val strategyGoalId: Long?,
            val isCampaignMigration: Boolean = false
        ) {

            var allGoalsAreAvailable = isCampaignMigration || (isRequestFromInternalNetwork
                && !availableFeatures.contains(FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED))

            var allowMeaningfulGoalValueFromMetrika: Boolean = availableFeatures.contains(
                FeatureName.ALLOW_MEANINGFUL_GOAL_VALUE_FROM_METRIKA
            )

            fun isMeaningfulGoalsCheckRequired(): Boolean =
                STRATEGY_TYPES_TO_CHECK_MEANINGFUL_GOALS.contains(strategyType) &&
                    strategyGoalId == MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID
        }

        private val STRATEGY_TYPES_TO_CHECK_MEANINGFUL_GOALS =
            setOf(StrategyName.AUTOBUDGET_CRR, StrategyName.AUTOBUDGET_ROI, StrategyName.AUTOBUDGET)
    }
}
