package ru.yandex.direct.core.entity.strategy.type.withmetrikacounters;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.metrikacounter.model.StrategyMetrikaCounter;
import ru.yandex.direct.dbschema.ppc.enums.StrategyMetrikaCountersSource;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppc.tables.StrategyMetrikaCounters.STRATEGY_METRIKA_COUNTERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class StrategyMetrikaCountersRepository {
    public static final JooqMapperWithSupplier<StrategyMetrikaCounter> METRIKA_COUNTERS_MAPPER =
            createStrategyMetrikaCounterMapper();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public StrategyMetrikaCountersRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public Map<Long, List<StrategyMetrikaCounter>> getMetrikaCounters(int shard, Collection<Long> strategyIds) {
        return getMetrikaCounters(dslContextProvider.ppc(shard), strategyIds);
    }

    public Map<Long, List<StrategyMetrikaCounter>> getMetrikaCounters(DSLContext context,
                                                                      Collection<Long> strategyIds) {
        return context
                .select(METRIKA_COUNTERS_MAPPER.getFieldsToRead())
                .from(STRATEGY_METRIKA_COUNTERS)
                .where(STRATEGY_METRIKA_COUNTERS.STRATEGY_ID.in(strategyIds))
                .fetchGroups(STRATEGY_METRIKA_COUNTERS.STRATEGY_ID, METRIKA_COUNTERS_MAPPER::fromDb);
    }

    private static JooqMapperWithSupplier<StrategyMetrikaCounter> createStrategyMetrikaCounterMapper() {
        return JooqMapperWithSupplierBuilder.builder(StrategyMetrikaCounter::new)
                .map(property(StrategyMetrikaCounter.ID,
                        STRATEGY_METRIKA_COUNTERS.METRIKA_COUNTER))
                .map(property(StrategyMetrikaCounter.STRATEGY_ID,
                        STRATEGY_METRIKA_COUNTERS.STRATEGY_ID))
                .map(convertibleProperty(StrategyMetrikaCounter.SOURCE,
                        STRATEGY_METRIKA_COUNTERS.SOURCE, StrategyMetrikaCountersRepository::toCounterSource,
                        StrategyMetrikaCountersRepository::fromCounterSource))
                .map(booleanProperty(StrategyMetrikaCounter.HAS_ECOMMERCE,
                        STRATEGY_METRIKA_COUNTERS.HAS_ECOMMERCE))
                .map(booleanProperty(StrategyMetrikaCounter.IS_DELETED,
                        STRATEGY_METRIKA_COUNTERS.IS_DELETED))
                .build();
    }

    private static StrategyMetrikaCountersSource fromCounterSource(@Nullable MetrikaCounterSource source) {
        if (source == null) {
            return StrategyMetrikaCountersSource.unknown;
        }
        return StrategyMetrikaCountersSource.valueOf(source.name().toLowerCase());
    }

    private static MetrikaCounterSource toCounterSource(@Nullable StrategyMetrikaCountersSource source) {
        if (source == null) {
            return MetrikaCounterSource.UNKNOWN;
        } else {
            return MetrikaCounterSource.valueOf(source.name().toUpperCase());
        }
    }
}
