package ru.yandex.direct.core.entity.strategy.type.withmetrikacounters;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.jooq.TableField;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.metrikacounter.model.StrategyMetrikaCounter;
import ru.yandex.direct.core.entity.strategy.container.StrategyRepositoryContainer;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMetrikaCounters;
import ru.yandex.direct.core.entity.strategy.repository.AbstractStrategyMultiRowEntityRepositoryTypeSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.repository.container.AddOrUpdateAndDeleteByKeyContainer;
import ru.yandex.direct.multitype.repository.container.AddOrUpdateAndDeleteContainer;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toMetrikaCounterSource;
import static ru.yandex.direct.core.entity.strategy.type.withmetrikacounters.StrategyMetrikaCountersRepository.METRIKA_COUNTERS_MAPPER;
import static ru.yandex.direct.dbschema.ppc.Tables.STRATEGY_METRIKA_COUNTERS;
import static ru.yandex.direct.model.AppliedChanges.isChanged;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class StrategyWithMetrikaCountersRepositoryTypeSupport
        extends AbstractStrategyMultiRowEntityRepositoryTypeSupport<StrategyWithMetrikaCounters,
        StrategyMetrikaCounter> {

    private final StrategyMetrikaCountersRepository strategyMetrikaCountersRepository;

    @Autowired
    protected StrategyWithMetrikaCountersRepositoryTypeSupport(DslContextProvider dslContextProvider,
                                                               StrategyMetrikaCountersRepository strategyMetrikaCountersRepository) {
        super(dslContextProvider);
        this.strategyMetrikaCountersRepository = strategyMetrikaCountersRepository;
    }

    @Override
    public JooqMapper<StrategyWithMetrikaCounters> createMapper() {
        return null;
    }

    @Override
    public Set<TableField<?, Long>> getPrimaryKeys() {
        return Set.of();
    }

    @Override
    protected AddOrUpdateAndDeleteContainer<StrategyMetrikaCounter> buildAddOrUpdateAndDeleteContainer(
            DSLContext context,
            StrategyRepositoryContainer updateParameters,
            Collection<AppliedChanges<StrategyWithMetrikaCounters>> strategiesChangesCollection) {
        var builder = AddOrUpdateAndDeleteByKeyContainer.builder(StrategyMetrikaCounter::getId)
                .skipNewRowsByEquals();

        List<Long> strategyWithChangedMetrikaCounterIds = filterAndMapList(strategiesChangesCollection,
                appliedChanges -> appliedChanges.changed(StrategyWithMetrikaCounters.METRIKA_COUNTERS),
                appliedChanges -> appliedChanges.getModel().getId());

        Map<Long, List<StrategyMetrikaCounter>> metrikaCountersByStrategyId =
                strategyMetrikaCountersRepository.getMetrikaCounters(updateParameters.getShard(),
                        strategyWithChangedMetrikaCounterIds);

        StreamEx.of(strategiesChangesCollection)
                .filter(isChanged(StrategyWithMetrikaCounters.METRIKA_COUNTERS))
                .forEach(ac -> builder.addDiff(
                        metrikaCountersByStrategyId.getOrDefault(ac.getModel().getId(), List.of()),
                        extractNewRepositoryModels(updateParameters, singletonList(ac.getModel()))));

        return builder.build();
    }

    @NotNull
    @Override
    protected List<StrategyMetrikaCounter> extractNewRepositoryModels(StrategyRepositoryContainer updateParameters,
                                                                      Collection<StrategyWithMetrikaCounters> strategies) {
        var userCountersExtendedById = updateParameters.getUserCountersExtendedById();
        return StreamEx.of(strategies)
                .mapToEntry(BaseStrategy::getId, StrategyWithMetrikaCounters::getMetrikaCounters)
                .nonNullValues()
                .flatMapValues(Collection::stream)
                .mapKeyValue((strategyId, counterId) ->
                        getStrategyMetrikaCounter(userCountersExtendedById.get(counterId), strategyId, counterId))
                .toList();
    }

    private static StrategyMetrikaCounter getStrategyMetrikaCounter(
            @Nullable CounterInfoDirect userCounter, Long strategyId, Long counterId) {
        if (userCounter == null) {
            return new StrategyMetrikaCounter()
                    .withId(counterId)
                    .withIsDeleted(false)
                    .withStrategyId(strategyId);
        }
        return new StrategyMetrikaCounter()
                .withId(counterId)
                .withStrategyId(strategyId)
                .withIsDeleted(false)
                .withHasEcommerce(userCounter.getEcommerce())
                .withSource(toMetrikaCounterSource(userCounter.getCounterSource()));
    }

    @Override
    protected void delete(DSLContext context, Collection<StrategyMetrikaCounter> repositoryModelsToDelete) {
        Set<Long> counterIdsToDelete = listToSet(repositoryModelsToDelete, StrategyMetrikaCounter::getId);
        Set<Long> strategyIdsToDelete = listToSet(repositoryModelsToDelete, StrategyMetrikaCounter::getStrategyId);
        context.deleteFrom(STRATEGY_METRIKA_COUNTERS)
                .where(STRATEGY_METRIKA_COUNTERS.STRATEGY_ID.in(strategyIdsToDelete)
                        .and(STRATEGY_METRIKA_COUNTERS.METRIKA_COUNTER.in(counterIdsToDelete)))
                .execute();
    }

    @Override
    protected void addOrUpdate(DSLContext context, Collection<StrategyMetrikaCounter> repositoryModelsToAddOrUpdate) {
        new InsertHelper<>(context, STRATEGY_METRIKA_COUNTERS)
                .addAll(METRIKA_COUNTERS_MAPPER, repositoryModelsToAddOrUpdate)
                .onDuplicateKeyUpdate()
                .set(STRATEGY_METRIKA_COUNTERS.HAS_ECOMMERCE, MySQLDSL.values(STRATEGY_METRIKA_COUNTERS.HAS_ECOMMERCE))
                .executeIfRecordsAdded();
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<StrategyWithMetrikaCounters> models) {
        var strategyIds = listToSet(models, BaseStrategy::getId);
        Map<Long, List<StrategyMetrikaCounter>> metrikaCountersByStrategyId =
                strategyMetrikaCountersRepository.getMetrikaCounters(dslContext, strategyIds);

        Map<Long, List<Long>> metrikaCounterIdsByStrategyId = EntryStream.of(metrikaCountersByStrategyId)
                .mapValues(counters -> mapList(counters, StrategyMetrikaCounter::getId))
                .toMap();

        models.forEach(strategy -> strategy.withMetrikaCounters(
                metrikaCounterIdsByStrategyId.getOrDefault(strategy.getId(), List.of())));
    }

    @Override
    public Class<StrategyWithMetrikaCounters> getTypeClass() {
        return StrategyWithMetrikaCounters.class;
    }
}
