package ru.yandex.direct.core.entity.strategy.type.withmetrikacounters

import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CollectionDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import javax.annotation.ParametersAreNonnullByDefault

@ParametersAreNonnullByDefault
class StrategyWithMetrikaCountersValidator(private val container: ValidationContainer) :
    Validator<List<Long?>?, Defect<*>> {
    override fun apply(metrikaCounters: List<Long?>?): ValidationResult<List<Long?>?, Defect<*>> {

        val vb = ListValidationBuilder.of<Long, Defect<*>>(metrikaCounters)

        vb
            .check(CommonConstraints.notNull(), When.isTrue(container.isCounterIsRequiredForStrategy()))

        if (metrikaCounters != null) {
            vb.checkEach(CommonConstraints.notNull())
                .checkEach(CommonConstraints.validId())
                .checkEach(CollectionConstraints.unique())
                .check(CollectionConstraints.minListSize(container.getMinCountersCount()))
                .check(maxClientMetrikaCountersCountOnStrategy(container.getMaxCountersCount()))
                //todo: поменять дефект
                .checkEach(
                    CommonConstraints.inSet(container.availableCounterIds),
                    CampaignDefects.metrikaCounterIsUnavailable(),
                    When.isTrue(container.isCounterIsRequiredForStrategy())
                )
        }
        return vb.result
    }

    data class ValidationContainer(
        val campaignType: CampaignType?,
        val availableCounterIds: Set<Long>,
        val systemCountersIds: Set<Long>,
        val isCampaignToStrategyMigrationOneshot: Boolean = false
    ) {

        fun getMinCountersCount() = if (isCounterIsRequiredForStrategy()) 1 else 0

        fun getMaxCountersCount() = if (isCounterIsRequiredForStrategy()) 1 else 100

        fun isCounterIsRequiredForStrategy() =
            CampaignType.PERFORMANCE == campaignType && !isCampaignToStrategyMigrationOneshot
    }

    private fun maxClientMetrikaCountersCountOnStrategy(maxCountersCount: Int): Constraint<List<Long?>, Defect<*>> {
        return Constraint.fromPredicate(
            { counters: List<Long?> ->
                counters.count { !container.systemCountersIds.contains(it) } <= maxCountersCount
            },
            CollectionDefects.maxCollectionSize(maxCountersCount)
        )
    }
}
