package ru.yandex.direct.core.entity.strategy.type.withmetrikacounters

import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints
import ru.yandex.direct.core.entity.strategy.container.AbstractStrategyOperationContainer
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMetrikaCounters
import ru.yandex.direct.core.entity.strategy.validation.AbstractStrategyValidatorProvider
import ru.yandex.direct.metrika.client.MetrikaClientException
import ru.yandex.direct.metrika.client.model.response.UserCounters
import ru.yandex.direct.utils.InterruptedRuntimeException
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder
import javax.annotation.ParametersAreNonnullByDefault

@ParametersAreNonnullByDefault
object StrategyWithMetrikaCountersValidatorProvider : AbstractStrategyValidatorProvider<StrategyWithMetrikaCounters>() {
    private val logger = LoggerFactory.getLogger(StrategyWithMetrikaCountersValidatorProvider::class.java)

    override fun createStrategyValidator(container: AbstractStrategyOperationContainer):
        Validator<StrategyWithMetrikaCounters, Defect<*>> {

        val availableCounterIds: Set<Long>
        try {
            availableCounterIds = if (!container.options.isCampaignToPackageStrategyOneshot) {
                getAvailableCounterIds(container)
            } else {
                emptySet()
            }
        } catch (e: MetrikaClientException) {
            logger.warn(
                "Got an exception when querying available counters from metrika by clientId {}.",
                container.clientId, e
            )
            return METRIKA_RETURNS_RESULT_WITH_ERRORS_VALIDATOR
        } catch (e: InterruptedRuntimeException) {
            logger.warn(
                "Got an exception when querying available counters from metrika by clientId {}.",
                container.clientId, e
            )
            return METRIKA_RETURNS_RESULT_WITH_ERRORS_VALIDATOR
        }

        val clientSystemCounterIds: Set<Long>
        try {
            clientSystemCounterIds = if (!container.options.isCampaignToPackageStrategyOneshot) {
                getClientSystemCounterIds(
                    container
                )
            } else {
                emptySet()
            }
        } catch (e: MetrikaClientException) {
            logger.warn(
                "Got an exception when querying for organization counters from metrika by clientId {}. " +
                    "Exception: {}", container.clientId, e
            )
            return METRIKA_RETURNS_RESULT_WITH_ERRORS_VALIDATOR
        } catch (e: InterruptedRuntimeException) {
            logger.warn(
                "Got an exception when querying for organization counters from metrika by clientId {}. " +
                    "Exception: {}", container.clientId, e
            )
            return METRIKA_RETURNS_RESULT_WITH_ERRORS_VALIDATOR
        }

        return Validator { strategy: StrategyWithMetrikaCounters ->
            val strategyWithMetrikaCountersValidator = StrategyWithMetrikaCountersValidator(
                StrategyWithMetrikaCountersValidator.ValidationContainer(
                    container.campaignType(strategy),
                    availableCounterIds,
                    clientSystemCounterIds,
                    container.options.isCampaignToPackageStrategyOneshot
                )
            )
            val vb = ModelItemValidationBuilder.of(strategy)
            vb.item(StrategyWithMetrikaCounters.METRIKA_COUNTERS)
                .checkBy(strategyWithMetrikaCountersValidator)

            vb.result
        }
    }

    private fun getAvailableCounterIds(container: AbstractStrategyOperationContainer): Set<Long> {
        return container.metrikaClientAdapter!!.usersCountersNumByCampaignCounterIds
            .flatMap { obj: UserCounters -> obj.counterIds }
            .map { obj: Int -> obj.toLong() }
            .toSet()
    }

    private val METRIKA_RETURNS_RESULT_WITH_ERRORS_VALIDATOR = Validator { strategy: StrategyWithMetrikaCounters? ->
        val ivb = ModelItemValidationBuilder.of(strategy)
        ivb.check(CampaignConstraints.metrikaReturnsResultWithErrors(), When.notNull())
        ivb.result
    }

    private fun getClientSystemCounterIds(container: AbstractStrategyOperationContainer): Set<Long> {
        return container.metrikaClientAdapter!!.usersCountersNumExtendedByCampaignCounterIds
            .flatMap { it.counters }
            .filter { CampaignConverter.SPRAV == it.counterSource }
            .map { it.id }
            .map { java.lang.Long.valueOf(it.toLong()) }
            .toSet()
    }
}
