package ru.yandex.direct.core.entity.strategy.type.withpayforconversion

import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BY_ALL_GOALS_GOAL_ID
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TYPES_AVAILABLE_FOR_PAY_FOR_CONVERSION_EXTENDED_MODE
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects
import ru.yandex.direct.core.entity.strategy.model.StrategyName
import ru.yandex.direct.core.entity.strategy.validation.StrategyValidationUtils.campaignTypeNullOr
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.utils.CommonUtils.nvl
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.validateObject

class PayForConversionValidator(private val container: ValidationContainer) : Validator<Boolean, Defect<*>> {
    override fun apply(t: Boolean?) =
        validateObject(t) {
            val isPayForConversionEnabled = nvl(t, false)
            checkBy({ isEnabled -> this@PayForConversionValidator.checkSpecialGoals(isEnabled, container.goalId) }, When.isTrue(container.shouldValidatePayForConversion))
            checkBy(
                this@PayForConversionValidator::validateStrategyConsistentWhenPayForConversionExtendedModeEnabled,
                When.isTrue(isPayForConversionEnabled && STRATEGY_TYPES_FOR_EXTENDED_MODE_VALIDATION.contains(container.strategyType)))
        }

    companion object {
        data class ValidationContainer(val strategyType: StrategyName,
                                       val goalId: Long?,
                                       val campaignType: CampaignType?,
                                       val availableFeatures: Set<FeatureName>,
                                       val isValidatePayForConversionInAutobudgetAvgCpaPerFilter: Boolean) {
            val shouldValidatePayForConversion: Boolean =
                isValidatePayForConversionInAutobudgetAvgCpaPerFilter || strategyType != StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER

            val campaignTypeAvailableForPayForConversion = campaignTypeNullOr(campaignType, CAMPAIGN_TYPES_AVAILABLE_FOR_PAY_FOR_CONVERSION_EXTENDED_MODE)

            val isMeaningfulGoalsWithPayForConversionAllowed =
                availableFeatures.contains(FeatureName.ALLOW_ALL_MEANINGFUL_GOALS_FOR_PAY_FOR_CONVERSION_STRATEGIES) &&
                strategyType == StrategyName.AUTOBUDGET_CRR
        }
    }

    private fun checkSpecialGoals(isPayForConversionEnabled: Boolean?, goalId: Long?): ValidationResult<Boolean?, Defect<*>> =
        if ((BY_ALL_GOALS_GOAL_ID == goalId ||
                CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID == goalId &&
                !container.isMeaningfulGoalsWithPayForConversionAllowed) && isPayForConversionEnabled == true) {
            ValidationResult.failed(isPayForConversionEnabled, StrategyDefects.payForConversionDoesNotAllowAllGoals())
        } else {
            ValidationResult(isPayForConversionEnabled)
        }

    /**
     * Проверка возможности использовать оплату за конверсии
     */
    private fun validateStrategyConsistentWhenPayForConversionExtendedModeEnabled(payForConversion: Boolean?) = validateObject(payForConversion) {
        check(Constraint.fromPredicate(
            {
                (container.campaignTypeAvailableForPayForConversion
                    || container.campaignType == CampaignType.CPM_BANNER && container.availableFeatures
                    .contains(FeatureName.CPA_STRATEGY_IN_CPM_BANNER_CAMPAIGN_ENABLED))
            },
            StrategyDefects.inconsistentStrategyToCampaignType())
        )
    }

    private val STRATEGY_TYPES_FOR_EXTENDED_MODE_VALIDATION =
        setOf(StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP, StrategyName.AUTOBUDGET_AVG_CPA, StrategyName.AUTOBUDGET_AVG_CPI)

}
