package ru.yandex.direct.core.entity.strategy.utils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.IntStream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpaPerFilter;
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpcPerFilter;
import ru.yandex.direct.core.entity.strategy.model.AutobudgetAvgCpi;
import ru.yandex.direct.core.entity.strategy.model.AutobudgetCrr;
import ru.yandex.direct.core.entity.strategy.model.AutobudgetRoi;
import ru.yandex.direct.core.entity.strategy.model.AutobudgetWeekBundle;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy;
import ru.yandex.direct.core.entity.strategy.model.DefaultManualStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgBid;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgCpa;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgCpm;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithAvgCpv;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithBid;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithConversion;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithCustomPeriodBudget;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithCustomPeriodBudgetAndCustomBid;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithDayBudget;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithLastBidderRestartTime;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMeaningfulGoals;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMetrikaCounters;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithPayForConversion;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithWeeklyBudget;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.PROPERTIES_BY_TYPE;

@ParametersAreNonnullByDefault
public class StrategyComparingUtils {

    // изменения этих полей не значимы для изменения пакетной стратегии/они зависят не от значений полей в кампании
    private static final Set<ModelProperty<? extends Model, ?>> STRATEGY_PROPERTIES_TO_IGNORE_WHEN_COMPARING_STRATEGIES = Set.of(
            BaseStrategy.ID,
            CommonStrategy.NAME,
            CommonStrategy.CIDS,
            CommonStrategy.IS_PUBLIC,
            CommonStrategy.CLIENT_ID,
            CommonStrategy.WALLET_ID,
            CommonStrategy.LAST_CHANGE,
            CommonStrategy.STATUS_ARCHIVED,
            StrategyWithDayBudget.DAY_BUDGET_LAST_CHANGE,
            StrategyWithDayBudget.DAY_BUDGET_DAILY_CHANGE_COUNT,
            StrategyWithLastBidderRestartTime.LAST_BIDDER_RESTART_TIME,
            StrategyWithCustomPeriodBudgetAndCustomBid.LAST_UPDATE_TIME);

    private static final Comparator<Long> customLongComparator = longComparator();
    private static final Comparator<Boolean> customBooleanComparator = booleanNullAsFalseComparator();
    private static final Comparator<BigDecimal> customBigDecimalComparator = bigDecimalComparator();

    private static final Map<ModelProperty<? extends Model, Long>, Comparator<Long>> customComparatorsByPropsWithLongValues =
            StreamEx.<ModelProperty<? extends Model, Long>>of(AutobudgetCrr.CRR,
                    AutobudgetRoi.RESERVE_RETURN,
                    AutobudgetWeekBundle.LIMIT_CLICKS,
                    StrategyWithConversion.GOAL_ID,
                    StrategyWithCustomPeriodBudgetAndCustomBid.DAILY_CHANGE_COUNT)
                    .mapToEntry(property -> customLongComparator)
                    .toMap();

    private static final Map<ModelProperty<? extends Model, Boolean>, Comparator<Boolean>> customComparatorsByPropsWithBooleanValues =
            StreamEx.<ModelProperty<? extends Model, Boolean>>of(DefaultManualStrategy.ENABLE_CPC_HOLD,
                    StrategyWithCustomPeriodBudget.AUTO_PROLONGATION,
                    StrategyWithPayForConversion.IS_PAY_FOR_CONVERSION_ENABLED)
                    .mapToEntry(property -> customBooleanComparator)
                    .toMap();

    private static final Map<ModelProperty<? extends Model, BigDecimal>, Comparator<BigDecimal>> customComparatorsByPropsWithBigDecimalValues =
            StreamEx.<ModelProperty<? extends Model, BigDecimal>>of(AutobudgetAvgCpaPerFilter.FILTER_AVG_CPA,
                    AutobudgetAvgCpcPerFilter.FILTER_AVG_BID,
                    AutobudgetAvgCpi.AVG_CPI,
                    AutobudgetRoi.PROFITABILITY,
                    AutobudgetRoi.ROI_COEF,
                    StrategyWithAvgBid.AVG_BID,
                    StrategyWithAvgCpa.AVG_CPA,
                    StrategyWithAvgCpm.AVG_CPM,
                    StrategyWithAvgCpv.AVG_CPV,
                    StrategyWithBid.BID,
                    StrategyWithCustomPeriodBudget.BUDGET,
                    StrategyWithDayBudget.DAY_BUDGET,
                    StrategyWithWeeklyBudget.SUM)
                    .mapToEntry(property -> customBigDecimalComparator)
                    .toMap();

    private static final Map<ModelProperty<? extends Model, ?>, Comparator<?>> customComparatorsByPropsWithOtherTypesValues =
            Map.of(StrategyWithMetrikaCounters.METRIKA_COUNTERS, metrikaCountersComparator(),
                    StrategyWithMeaningfulGoals.MEANINGFUL_GOALS, meaningfulGoalsComparator());

    private static final Map<ModelProperty<? extends Model, ?>, Comparator<?>> customComparatorsByProps =
            StreamEx.of(customComparatorsByPropsWithLongValues, customComparatorsByPropsWithBooleanValues,
                    customComparatorsByPropsWithBigDecimalValues, customComparatorsByPropsWithOtherTypesValues)
                    .map(Map::entrySet)
                    .flatMap(Set::stream)
                    .toMap(Map.Entry::getKey, Map.Entry::getValue);

    public static boolean areDifferentStrategies(BaseStrategy oldStrategy,
                                                 BaseStrategy newStrategy) {
        if (newStrategy.getClass().equals(oldStrategy.getClass())) {
            return !StreamEx.of(PROPERTIES_BY_TYPE.get(oldStrategy.getType()))
                    .filter(property -> !STRATEGY_PROPERTIES_TO_IGNORE_WHEN_COMPARING_STRATEGIES.contains(property))
                    .allMatch(property -> areEqualStrategyPropertyValues(newStrategy, oldStrategy, property));
        }

        return true;
    }

    private static <T extends Model, V> boolean areEqualStrategyPropertyValues(BaseStrategy firstStrategy,
                                                                               BaseStrategy secondStrategy,
                                                                               ModelProperty<T, V> property) {
        V firstStrategyValue = property.get((T) firstStrategy);
        V secondStrategyValue = property.get((T) secondStrategy);

        return arePropertyValuesEquals(
                property,
                firstStrategyValue,
                secondStrategyValue
        );
    }

    public static <T extends Model, V> boolean arePropertyValuesEquals(
            ModelProperty<T, V> property,
            @Nullable V value1,
            @Nullable V value2
    ) {
        if (customComparatorsByProps.containsKey(property)) {
            var comparator = (Comparator<V>) customComparatorsByProps.get(property);
            return comparator.compare(value1, value2) == 0;
        } else if (value1 == value2) {
            return true;
        } else if (value1 == null || value2 == null) {
            return false;
        } else {
            return value1.equals(value2);
        }

    }

    private static Comparator<Long> longComparator() {
        return StrategyComparingUtils::compareNullableValues;
    }

    private static Comparator<BigDecimal> bigDecimalComparator() {
        return StrategyComparingUtils::compareNullableValues;
    }

    private static Comparator<Boolean> booleanNullAsFalseComparator() {
        return StrategyComparingUtils::compareBooleansNullAsFalse;
    }

    private static Comparator<List<Long>> metrikaCountersComparator() {
        return (l1, l2) -> compareNullableLists(l1, l2, longComparator());
    }

    private static Comparator<List<MeaningfulGoal>> meaningfulGoalsComparator() {
        return (l1, l2) -> compareNullableLists(l1, l2, meaningfulGoalComparator());
    }

    private static Comparator<MeaningfulGoal> meaningfulGoalComparator() {
        return (v1, v2) -> {
            if (Objects.equals(v1, v2)) {
                return 0;
            } else if (v1 == null) {
                return -1;
            } else if (v2 == null) {
                return 1;
            }

            return Comparator.comparing(MeaningfulGoal::getConversionValue, bigDecimalComparator())
                    .thenComparing(MeaningfulGoal::getGoalId, longComparator())
                    .thenComparing(MeaningfulGoal::getIsMetrikaSourceOfValue, booleanNullAsFalseComparator())
                    .compare(v1, v2);
        };
    }

    private static <V> int compareNullableLists(@Nullable List<V> list1,
                                                @Nullable List<V> list2,
                                                Comparator<V> elementComparator) {
        if (Objects.equals(list1, list2)) {
            return 0;
        } else if (list1 == null) {
            return -1;
        } else if (list2 == null) {
            return 1;
        }

        var firstStrategyGoals = new ArrayList<>(list1);
        var secondStrategyGoals = new ArrayList<>(list2);

        return compareLists(firstStrategyGoals, secondStrategyGoals, elementComparator);
    }

    private static <V> int compareLists(List<V> list1, List<V> list2, Comparator<V> comparator) {
        if (list1.size() != list2.size()) {
            return list1.size() < list2.size() ? -1 : 1;
        }

        list1.sort(comparator);
        list2.sort(comparator);

        var firstNotEqualId = IntStream.range(0, list1.size())
                .dropWhile(i -> comparator.compare(list1.get(i), list2.get(i)) == 0)
                .findFirst();
        if (firstNotEqualId.isPresent()) {
            int id = firstNotEqualId.getAsInt();
            return comparator.compare(list1.get(id), list2.get(id));
        } else {
            return 0;
        }
    }

    private static int compareBooleansNullAsFalse(@Nullable Boolean v1, @Nullable Boolean v2) {
        Set<Boolean> equalValues = new HashSet<>();
        equalValues.add(null);
        equalValues.add(false);

        boolean areFalse = equalValues.contains(v1) && equalValues.contains(v2);
        boolean areTrue = v1 != null && v2 != null && v1 && v2;
        if (areFalse || areTrue) {
            return 0;
        } else if (v1 != null && v1) {
            return 1;
        } else {
            return -1;
        }
    }

    private static <V extends Comparable<V>> int compareNullableValues(@Nullable V v1, @Nullable V v2) {
        if (Objects.equals(v1, v2)) {
            return 0;
        } else if (v1 == null) {
            return -1;
        } else if (v2 == null) {
            return 1;
        } else {
            return v1.compareTo(v2);
        }
    }

}
