package ru.yandex.direct.core.entity.strategy.validation.update

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyArchivingNotAllowed
import ru.yandex.direct.core.entity.strategy.container.StrategyRepositoryContainer
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants
import ru.yandex.direct.core.entity.strategy.utils.StrategyModelUtils.campaignIds
import ru.yandex.direct.core.entity.strategy.utils.StrategyModelUtils.clientId
import ru.yandex.direct.dbschema.ppc.tables.Strategies
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

@Service
class StrategyArchiveValidationService(
    private val strategyTypedRepository: StrategyTypedRepository
) {

    fun validateStrategyArchiveOperation(
        strategyRepositoryContainer: StrategyRepositoryContainer,
        modelChanges: List<ModelChanges<CommonStrategy>>
    ): ValidationResult<List<ModelChanges<CommonStrategy>>, Defect<*>> {
        val strategiesById =
            strategyTypedRepository.getIdToModelTyped(strategyRepositoryContainer.shard, modelChanges.map { it.id })

        return ListValidationBuilder.of<ModelChanges<CommonStrategy>, Defect<*>>(modelChanges)
            .checkEach(validateStrategyExists(strategiesById.keys))
            .checkEach(
                validateAccessForClientId(strategyRepositoryContainer.clientId.asLong(), strategiesById),
                When.isValid()
            )
            .checkEach(validateExistingCampaigns(strategiesById), When.isValid())
            .result
    }

    fun validateStrategyUnarchiveOperation(
        strategyRepositoryContainer: StrategyRepositoryContainer,
        modelChanges: List<ModelChanges<CommonStrategy>>
    ): ValidationResult<List<ModelChanges<CommonStrategy>>, Defect<*>> {
        val strategiesById =
            strategyTypedRepository.getIdToModelTyped(strategyRepositoryContainer.shard, modelChanges.map { it.id })

        val existingClientStrategies = getExistingPublicClientStrategies(
            strategyRepositoryContainer.shard,
            strategyRepositoryContainer.clientId
        ).associateBy { it.id }

        val existingUnarchivedClientStrategiesNumber = existingClientStrategies.filter { (_, strategy) -> !strategy.statusArchived }.size

        val allClientUnarchivedStrategiesNumber = existingUnarchivedClientStrategiesNumber + modelChanges.filter {
            existingClientStrategies[it.id]?.statusArchived == true
        }.size

        return ListValidationBuilder.of<ModelChanges<CommonStrategy>, Defect<*>>(modelChanges)
            .check(
                Constraint.fromPredicate(
                    {
                        allClientUnarchivedStrategiesNumber <= StrategyConstants.MAX_UNARCHIVED_STRATEGIES_FOR_CLIENT_NUMBER
                    },
                    StrategyDefects.unarchivedStrategiesNumberLimitExceeded(StrategyConstants.MAX_UNARCHIVED_STRATEGIES_FOR_CLIENT_NUMBER)
                )
            )
            .checkEach(validateStrategyExists(strategiesById.keys))
            .checkEach(
                validateAccessForClientId(strategyRepositoryContainer.clientId.asLong(), strategiesById),
                When.isValid()
            )
            .result
    }

    private fun validateStrategyExists(existingStrategyIds: Set<Long>) =
        Constraint.fromPredicate({ changes: ModelChanges<CommonStrategy> ->
            existingStrategyIds.contains(changes.id)
        }, objectNotFound() as Defect<*>)

    private fun validateExistingCampaigns(strategiesById: Map<Long, BaseStrategy>): Constraint<ModelChanges<CommonStrategy>, Defect<*>> {
        val strategyIdToCampaignsId = strategiesById.mapValues { it.value.campaignIds() }

        return Constraint.fromPredicate({ changes: ModelChanges<CommonStrategy> ->
            strategyIdToCampaignsId[changes.id].isNullOrEmpty()
        }, strategyArchivingNotAllowed())
    }

    private fun validateAccessForClientId(
        clientId: Long,
        strategiesById: Map<Long, BaseStrategy>
    ): Constraint<ModelChanges<CommonStrategy>, Defect<*>> {
        val strategyIdToClientId = strategiesById.mapValues { it.value.clientId() }

        return Constraint.fromPredicate({ changes: ModelChanges<CommonStrategy> ->
            clientId == strategyIdToClientId[changes.id]
        }, objectNotFound())
    }

    private fun getExistingPublicClientStrategies(shard: Int, clientId: ClientId): List<CommonStrategy> {
        return strategyTypedRepository.getSafely(
            shard,
            ConditionFilterFactory.whereEqFilter(Strategies.STRATEGIES.CLIENT_ID, clientId.asLong()),
            CommonStrategy::class.java
        ).filter { it.isPublic }
    }
}
