package ru.yandex.direct.core.entity.strategy.validation.validators

import ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.weekBudgetLessThan
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy.StrategyValidatorConstants
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan
import ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder
import java.math.BigDecimal

class FilterAvgCpaOrAvgCpiValidator(private val container: ValidationContainer) : Validator<BigDecimal, Defect<*>> {

    override fun apply(t: BigDecimal?): ValidationResult<BigDecimal, Defect<*>> {
        return ModelItemValidationBuilder.of<BigDecimal, Defect<*>>(t)
            .checkBy(this::validateAvgCpaOrCpi)
            .result
    }

    private fun validateAvgCpaOrCpi(averageCpa: BigDecimal): ValidationResult<BigDecimal, Defect<*>> {
        val autobudgetPayForConversionAvgCpaWarning = CampaignWithStrategyValidationUtils.getAutobudgetPayForConversionAvgCpaWarning(
            container.availableFeatureNames,
            container.currency
        )
        return ModelItemValidationBuilder.of<BigDecimal, Defect<*>>(averageCpa)
            .check(notNull())
            .check(notLessThan(container.constants.minAvgCpa))
            .check(notGreaterThan(autobudgetPayForConversionAvgCpaWarning), When.isTrue(container.isPayForConversionEnabled))
            .check(notGreaterThan(container.currency.getAutobudgetAvgCpaWarning()), When.isFalse(container.isPayForConversionEnabled))
            .check(notGreaterThan(container.sum), weekBudgetLessThan(), When.isTrue(container.sum != null))
            .result
    }

    companion object {
        data class ValidationContainer(val currency: Currency,
                                       val isPayForConversionEnabled: Boolean,
                                       val availableFeatures: Set<FeatureName>,
                                       val constants: StrategyValidatorConstants,
                                       val sum: BigDecimal?) {
            val availableFeatureNames = availableFeatures.map(FeatureName::getName).toSet()
        }
    }

}
