package ru.yandex.direct.core.entity.tag.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.tag.model.CampaignTag;
import ru.yandex.direct.core.entity.tag.model.Tag;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.TAG_CAMPAIGN_LIST;
import static ru.yandex.direct.dbschema.ppc.Tables.TAG_GROUP;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Repository
@ParametersAreNonnullByDefault
public class TagRepository {

    private static final Field<Integer> USES_COUNT_FIELD = DSL.count(TAG_GROUP.PID).as("uses_count");

    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<Tag> tagMapper;
    private final JooqMapperWithSupplier<CampaignTag> campaignTagMapper;

    @Autowired
    public TagRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.tagMapper = JooqMapperWithSupplierBuilder.builder(Tag::new)
                .map(property(Tag.ID, TAG_CAMPAIGN_LIST.TAG_ID))
                .map(property(Tag.CAMPAIGN_ID, TAG_CAMPAIGN_LIST.CID))
                .map(property(Tag.NAME, TAG_CAMPAIGN_LIST.TAG_NAME))
                .map(property(Tag.CREATE_TIME, TAG_CAMPAIGN_LIST.CREATETIME))
                .build();
        this.campaignTagMapper = JooqMapperWithSupplierBuilder.builder(CampaignTag::new)
                .map(property(CampaignTag.ID, TAG_CAMPAIGN_LIST.TAG_ID))
                .map(property(CampaignTag.CAMPAIGN_ID, TAG_CAMPAIGN_LIST.CID))
                .map(property(CampaignTag.NAME, TAG_CAMPAIGN_LIST.TAG_NAME))
                .map(property(CampaignTag.CREATE_TIME, TAG_CAMPAIGN_LIST.CREATETIME))
                .readProperty(CampaignTag.USES_COUNT, fromField(USES_COUNT_FIELD))
                .build();
    }

    public JooqMapperWithSupplier<Tag> getTagMapper() {
        return tagMapper;
    }

    /**
     * Возвращает id тегов с привязкой к кампании
     *
     * @return маппинг id кампании на список id тегов
     */
    public Map<Long, List<Long>> getCampaignsTagIds(int shard, ClientId clientId, Set<Long> tagIds) {
        return dslContextProvider.ppc(shard)
                .select(TAG_CAMPAIGN_LIST.CID, TAG_CAMPAIGN_LIST.TAG_ID)
                .from(TAG_CAMPAIGN_LIST)
                .join(CAMPAIGNS).on(TAG_CAMPAIGN_LIST.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()).and(TAG_CAMPAIGN_LIST.TAG_ID.in(tagIds)))
                .fetchGroups(TAG_CAMPAIGN_LIST.CID, TAG_CAMPAIGN_LIST.TAG_ID);
    }

    /**
     * Возвращает список тегов с привязкой к группе
     *
     * @return маппинг id кампании на список тегов
     */
    public Map<Long, List<Tag>> getAdGroupsTags(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(tagMapper.getFieldsToRead())
                .select(TAG_GROUP.PID)
                .from(TAG_GROUP)
                .join(TAG_CAMPAIGN_LIST).using(TAG_CAMPAIGN_LIST.TAG_ID)
                .where(TAG_GROUP.PID.in(adGroupIds))
                .fetchGroups(TAG_GROUP.PID, tagMapper::fromDb);
    }

    public List<CampaignTag> getCampaignTagsWithUseCount(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyList();
        }

        return dslContextProvider.ppc(shard)
                .select(tagMapper.getFieldsToRead())
                .select(USES_COUNT_FIELD)
                .from(TAG_CAMPAIGN_LIST)
                .leftJoin(TAG_GROUP).on(TAG_GROUP.TAG_ID.eq(TAG_CAMPAIGN_LIST.TAG_ID))
                .where(TAG_CAMPAIGN_LIST.CID.in(campaignIds))
                .groupBy(TAG_CAMPAIGN_LIST.TAG_ID)
                .fetch(campaignTagMapper::fromDb);
    }
}
