package ru.yandex.direct.core.entity.takeout.service;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.takeout.model.TakeoutJobParams;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.TAKEOUT_REQUEST;

@Service
@ParametersAreNonnullByDefault
public class TakeoutService {
    private static final Logger logger = LoggerFactory.getLogger(TakeoutService.class);

    private final DbQueueRepository dbQueueRepository;
    private final ShardHelper shardHelper;

    public TakeoutService(DbQueueRepository dbQueueRepository, ShardHelper shardHelper) {
        this.dbQueueRepository = dbQueueRepository;
        this.shardHelper = shardHelper;
    }

    /**
     * Поставить в очередь задание на выгрузку для переданного uid
     * если данного uid нет в базе, ставим job в рандомный шард с clientId = 1
     */
    public void requestData(Long uid, String jobId, @Nullable Long unixTime) {
        TakeoutJobParams params = new TakeoutJobParams()
                .withUid(uid)
                .withJobId(jobId)
                .withUnixTime(unixTime);

        int shard;
        ClientId clientId = ClientId.fromLong(1);
        if (shardHelper.isExistentUid(uid)) {
            shard = shardHelper.getShardByUserId(uid);
            clientId = ClientId.fromLong(shardHelper.getClientIdByUid(uid));
        } else {
            logger.info("client for uid {} does not exist", uid);
            shard = shardHelper.getRandomShard();
        }
        dbQueueRepository.insertJob(shard, TAKEOUT_REQUEST, clientId, uid, params);
        logger.info("job queued in shard {} for clientId {}", shard, clientId);
    }
}
