package ru.yandex.direct.core.entity.targettag.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.targettags.model.TargetTag;
import ru.yandex.direct.dbschema.ppcdict.tables.records.TargetTagsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.TargetTags.TARGET_TAGS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Репозиторий для работы с информацией о таргет тегах {@link TargetTag}
 */
@Repository
@ParametersAreNonnullByDefault
public class TargetTagRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<TargetTag> targetTagMapper;

    @Autowired
    public TargetTagRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.targetTagMapper = createTargetTagMapper();
    }

    /**
     * Возвращает список всех таргет тегов
     */
    public List<TargetTag> getAllTargetTags() {
        return dslContextProvider.ppcdict()
                .select(targetTagMapper.getFieldsToRead())
                .from(TARGET_TAGS)
                .fetch(targetTagMapper::fromDb);
    }

    /**
     * Добавление новых и изменение существующих записей в таблице TARGET_TAGS
     */
    public int updateTargetTags(Collection<TargetTag> targetTags) {
        if (targetTags.isEmpty()) {
            return 0;
        }

        InsertHelper<TargetTagsRecord> helper = new InsertHelper<>(dslContextProvider.ppcdict(), TARGET_TAGS);
        helper.addAll(targetTagMapper, targetTags);

        if (helper.hasAddedRecords()) {
            helper.onDuplicateKeyUpdate()
                    .set(TARGET_TAGS.TAG_ID, MySQLDSL.values(TARGET_TAGS.TAG_ID))
                    .set(TARGET_TAGS.NAME, MySQLDSL.values(TARGET_TAGS.NAME))
                    .set(TARGET_TAGS.DESCRIPTION, MySQLDSL.values(TARGET_TAGS.DESCRIPTION));
        }
         return helper.executeIfRecordsAdded();
    }

    /**
     * Поиск таргет тегов по префиксу названия
     */
    public List<TargetTag> findTargetTagsByNamePrefix(String targetTagNamePrefix, int limit) {
        return dslContextProvider.ppcdict()
                .select(targetTagMapper.getFieldsToRead())
                .from(TARGET_TAGS)
                .where(TARGET_TAGS.NAME.startsWith(targetTagNamePrefix))
                .orderBy(TARGET_TAGS.NAME)
                .limit(limit)
                .fetch(targetTagMapper::fromDb);
    }

    private static JooqMapperWithSupplier<TargetTag> createTargetTagMapper() {
        return JooqMapperWithSupplierBuilder.builder(TargetTag::new)
                .map(property(TargetTag.ID, TARGET_TAGS.TAG_ID))
                .map(property(TargetTag.NAME, TARGET_TAGS.NAME))
                .map(property(TargetTag.DESCRIPTION, TARGET_TAGS.DESCRIPTION))
                .build();
    }
}
