package ru.yandex.direct.core.entity.tax.repository;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Select;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.tax.model.TaxInfo;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.grid.schema.yt.tables.TaxhistoryBs;
import ru.yandex.direct.ytcomponents.service.TaxHistoryDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static ru.yandex.direct.core.entity.statistics.service.OrderStatService.MILLION;
import static ru.yandex.direct.currency.Money.MONEY_MATH_CONTEXT;
import static ru.yandex.direct.grid.schema.yt.tables.TaxhistoryBs.TAXHISTORY_BS;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.longValueGetter;

@Repository
public class TaxHistoryRepository {

    private static final TaxhistoryBs HISTORY = TAXHISTORY_BS.as("H");
    private static final Field<Long> TAX_ID = aliased(HISTORY.TAX_ID);
    private static final Field<Long> START_DATE = aliased(HISTORY.START_DATE);
    private static final Field<Long> PERCENT = aliased(HISTORY.PERCENT);

    private final TaxHistoryDynContextProvider dynContextProvider;

    @Autowired
    public TaxHistoryRepository(TaxHistoryDynContextProvider dynContextProvider) {
        this.dynContextProvider = dynContextProvider;
    }

    /**
     * Получить информацию о налогах для TaxID in {@param taxIds}
     */
    public Map<Long, List<TaxInfo>> selectTaxInfos(Collection<Long> taxIds) {
        Select query = YtDSL.ytContext()
                .select(TAX_ID, START_DATE, PERCENT)
                .from(HISTORY)
                .where(TAX_ID.in(taxIds));

        UnversionedRowset rowset = dynContextProvider.getContext().executeSelect(query);
        TableSchema schema = rowset.getSchema();

        return StreamEx.of(rowset.getRows())
                .map(row -> composeTaxInfo(
                        longValueGetter(schema, TAX_ID).apply(row),
                        longValueGetter(schema, START_DATE).apply(row),
                        longValueGetter(schema, PERCENT).apply(row)
                ))
                .groupingBy(TaxInfo::getTaxId);
    }

    private TaxInfo composeTaxInfo(Long taxId, Long startDate, Long percent) {
        return new TaxInfo(taxId,
                Instant.ofEpochSecond(startDate).atZone(MSK).toLocalDate(),
                Percent.fromRatio(BigDecimal.valueOf(percent).divide(BigDecimal.valueOf(MILLION), MONEY_MATH_CONTEXT))
        );
    }
}
