package ru.yandex.direct.core.entity.tax.service;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.tax.model.TaxInfo;
import ru.yandex.direct.core.entity.tax.repository.TaxHistoryRepository;

import static ru.yandex.direct.core.entity.tax.service.TaxHistoryUtils.getTaxInfoForDay;

/**
 * Сервис для получения данных о НДС для статистики
 */
@Service
public class TaxHistoryService {

    private final TaxHistoryRepository taxHistoryRepository;

    private Cache<Long, List<TaxInfo>> taxInfoCache;

    @Autowired
    public TaxHistoryService(TaxHistoryRepository taxHistoryRepository) {
        this.taxHistoryRepository = taxHistoryRepository;

        this.taxInfoCache = CacheBuilder.newBuilder()
                .maximumSize(100)
                .expireAfterWrite(12, TimeUnit.HOURS)
                .build();
    }

    /**
     * Получить данные о налоге с TaxID {@param taxId} в день {@param day}
     */
    public TaxInfo getTaxInfo(Long taxId, LocalDate day) {

        List<TaxInfo> taxes = getTaxInfos(Set.of(taxId)).get(taxId);

        return getTaxInfoForDay(day, taxes);
    }

    /**
     * Получить данные о налогах с TaxID in {@param taxIds}
     */
    public Map<Long, List<TaxInfo>> getTaxInfos(Set<Long> taxIds) {

        Map<Long, List<TaxInfo>> taxInfoFromCache = EntryStream.of(taxInfoCache.asMap())
                .filterKeys(taxIds::contains)
                .toMap();

        if (taxInfoFromCache.size() != taxIds.size()) {
            Set<Long> taxIdsToSelect = StreamEx.of(taxIds)
                    .remove(taxInfoFromCache::containsKey)
                    .toSet();
            Map<Long, List<TaxInfo>> selectedTaxInfo = taxHistoryRepository.selectTaxInfos(taxIdsToSelect);

            if (selectedTaxInfo.isEmpty()) {
                throw new IllegalStateException("cannot find taxes info for TaxIDS: " +
                        StreamEx.of(taxIdsToSelect).joining());
            }

            taxInfoCache.putAll(selectedTaxInfo);

            return EntryStream.of(taxInfoFromCache)
                    .append(selectedTaxInfo)
                    .toMap();
        }

        return taxInfoFromCache;
    }

}
