package ru.yandex.direct.core.entity.timetarget.repository;

import java.time.ZoneId;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record;
import org.jooq.SelectConditionStep;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;
import ru.yandex.direct.core.entity.timetarget.model.GroupType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.dbschema.ppcdict.tables.GeoTimezones.GEO_TIMEZONES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.regions.Region.UKRAINE_REGION_ID;

@Repository
@ParametersAreNonnullByDefault
public class GeoTimezoneRepository {

    /**
     * В перл использовали мапу на "name", например "Europe/Moscow" -> "Москва"
     * Мы используем мапу на "countryId", потому что на country_id + timezone есть индекс в базе
     */
    private static final Map<String, Long> TIMEZONE_TO_COUNTRY_ID_MAP = Map.of(
            "Europe/Moscow", RUSSIA_REGION_ID,
            "Europe/Kiev", UKRAINE_REGION_ID
    );

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<GeoTimezone> jooqMapper;

    @Autowired
    public GeoTimezoneRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        jooqMapper = JooqMapperWithSupplierBuilder.builder(GeoTimezone::new)
                .map(property(GeoTimezone.TIMEZONE_ID, GEO_TIMEZONES.TIMEZONE_ID))
                .map(property(GeoTimezone.REGION_ID, GEO_TIMEZONES.COUNTRY_ID))
                .readProperty(GeoTimezone.TIMEZONE, fromField(GEO_TIMEZONES.TIMEZONE).by(ZoneId::of))
                .map(convertibleProperty(GeoTimezone.GROUP_TYPE, GEO_TIMEZONES.GROUP_NICK,
                        GroupType::fromTypedValue, GroupType::getTypedValue))
                .map(property(GeoTimezone.NAME_RU, GEO_TIMEZONES.NAME))
                .map(property(GeoTimezone.NAME_EN, GEO_TIMEZONES.ENAME))
                .map(property(GeoTimezone.NAME_UA, GEO_TIMEZONES.UA_NAME))
                .map(property(GeoTimezone.NAME_TR, GEO_TIMEZONES.TR_NAME))
                .build();
    }

    /**
     * @param timezoneIds Список ID timezone ({@link ru.yandex.direct.dbschema.ppcdict.tables.GeoTimezones#TIMEZONE_ID})
     * @return Список описаний таймзон.
     */
    public Collection<GeoTimezone> getGeoTimezonesByTimezoneIds(Collection<Long> timezoneIds) {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(GEO_TIMEZONES)
                .where(timezoneIds.isEmpty() ? GEO_TIMEZONES.TIMEZONE_ID.eq(GEO_TIMEZONES.TIMEZONE_ID)
                        : GEO_TIMEZONES.TIMEZONE_ID.in(timezoneIds))
                .fetch()
                .map(jooqMapper::fromDb);
    }

    /**
     * Список описаний всех таймзон.
     */
    public Map<Long, GeoTimezone> getAllTimezones() {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(GEO_TIMEZONES)
                .fetch()
                .map(jooqMapper::fromDb)
                .stream().collect(toMap(GeoTimezone::getTimezoneId, timezone -> timezone));
    }

    @Nullable
    public GeoTimezone getByCountryIdAndTimezone(Long countryId, ZoneId timezone) {
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(GEO_TIMEZONES)
                .where(GEO_TIMEZONES.COUNTRY_ID.eq(countryId)
                        .and(GEO_TIMEZONES.TIMEZONE.eq(timezone.getId())))
                .fetchOne(jooqMapper::fromDb);
    }

    /**
     * https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns/ConvertSubs
     * .pm?rev=r7981764#L110
     */
    public Map<String, GeoTimezone> getByTimeZones(Collection<String> timeZones) {
        Set<String> timeZonesNoCountry = new HashSet<>();
        Set<String> timeZonesWithCountry = new HashSet<>();

        for (String timeZone : timeZones) {
            if (TIMEZONE_TO_COUNTRY_ID_MAP.containsKey(timeZone)) {
                timeZonesWithCountry.add(timeZone);
            } else {
                timeZonesNoCountry.add(timeZone);
            }
        }

        SelectConditionStep<Record> step = dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(GEO_TIMEZONES)
                .where(GEO_TIMEZONES.TIMEZONE.in(timeZonesNoCountry));

        for (String timeZone : timeZonesWithCountry) {
            Long countryId = TIMEZONE_TO_COUNTRY_ID_MAP.get(timeZone);
            step = step.or(GEO_TIMEZONES.TIMEZONE.eq(timeZone).and(GEO_TIMEZONES.COUNTRY_ID.eq(countryId)));
        }

        List<GeoTimezone> fromDb = step
                .orderBy(GEO_TIMEZONES.GROUP_NICK.sortAsc("russia").nullsLast(), GEO_TIMEZONES.TIMEZONE_ID)
                .fetch(jooqMapper::fromDb);

        Map<String, GeoTimezone> result = new HashMap<>();
        fromDb.forEach(timeZone -> result.putIfAbsent(timeZone.getTimezone().getId(), timeZone));
        return result;
    }

}
