package ru.yandex.direct.core.entity.timetarget.repository;

import java.sql.Date;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.InsertValuesStep3;
import org.jooq.Record3;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.timetarget.model.HolidayItem;
import ru.yandex.direct.dbschema.ppcdict.enums.GreatHolidaysType;
import ru.yandex.direct.dbschema.ppcdict.tables.records.GreatHolidaysRecord;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.core.entity.timetarget.model.HolidayItem.Type.HOLIDAY;
import static ru.yandex.direct.dbschema.ppcdict.enums.GreatHolidaysType.holiday;
import static ru.yandex.direct.dbschema.ppcdict.enums.GreatHolidaysType.workday;
import static ru.yandex.direct.dbschema.ppcdict.tables.GreatHolidays.GREAT_HOLIDAYS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class ProductionCalendarRepository {

    private final DslContextProvider dslContextProvider;

    @Autowired
    public ProductionCalendarRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Для заданного года получает информацию о праздниках во всех известных регионах.
     * <p>
     * Используется для заполнения кэша в {@link ru.yandex.direct.core.entity.timetarget.service.ProductionCalendarProviderService}
     */
    public List<HolidayItem> getHolidaysByYear(int year) {
        LocalDate firstDay = LocalDate.ofYearDay(year, 1);
        LocalDate lastDay = firstDay.plus(1, ChronoUnit.YEARS);
        return dslContextProvider.ppcdict()
                .select(GREAT_HOLIDAYS.REGION_ID, GREAT_HOLIDAYS.HOLIDAY_DATE, GREAT_HOLIDAYS.TYPE)
                .from(GREAT_HOLIDAYS)
                .where(GREAT_HOLIDAYS.HOLIDAY_DATE.greaterOrEqual(firstDay))
                .and(GREAT_HOLIDAYS.HOLIDAY_DATE.lessThan(lastDay))
                .fetch().stream().map(this::toHolidayItem)
                .collect(Collectors.toList());
    }

    /**
     * Получает информацию о праздниках во всех известных регионах.
     * <p>
     */
    public List<HolidayItem> getAllHolidays() {
        return dslContextProvider.ppcdict()
                .select(GREAT_HOLIDAYS.REGION_ID, GREAT_HOLIDAYS.HOLIDAY_DATE, GREAT_HOLIDAYS.TYPE)
                .from(GREAT_HOLIDAYS)
                .fetch().stream().map(this::toHolidayItem)
                .collect(Collectors.toList());
    }

    private HolidayItem toHolidayItem(Record3<Long, LocalDate, GreatHolidaysType> r) {
        return new HolidayItem(
                r.value1().intValue(),
                r.value2(),
                HolidayItem.Type.valueOf(r.value3().getLiteral().toUpperCase()));
    }

    public int updateHolidays(final Collection<HolidayItem> holidayItems) {
        InsertValuesStep3<GreatHolidaysRecord, Long, LocalDate, GreatHolidaysType> step = dslContextProvider.ppcdict()
                .insertInto(GREAT_HOLIDAYS, GREAT_HOLIDAYS.REGION_ID, GREAT_HOLIDAYS.HOLIDAY_DATE, GREAT_HOLIDAYS.TYPE);
        for (HolidayItem hi : holidayItems) {
            final Field<Date> date =
                    null != hi.getDate() ? DSL.value(Date.valueOf(hi.getDate())) : SqlUtils.mysqlZeroDate();
            step = step.values(Arrays.asList(hi.getRegionId(), date, hi.getType() == HOLIDAY ? holiday : workday));
        }
        return step.onDuplicateKeyUpdate()
                .set(GREAT_HOLIDAYS.TYPE, MySQLDSL.values(GREAT_HOLIDAYS.TYPE))
                .execute();
    }

    public int deleteHolidays(Collection<HolidayItem> items) {
        return dslContextProvider.ppcdict()
                .deleteFrom(GREAT_HOLIDAYS)
                .where(row(GREAT_HOLIDAYS.REGION_ID, GREAT_HOLIDAYS.HOLIDAY_DATE, GREAT_HOLIDAYS.TYPE)
                        .in(mapList(items,
                                cb -> row(cb.getRegionId(), cb.getDate(),
                                        cb.getType() == HOLIDAY ? holiday : workday))))
                .execute();
    }
}
