package ru.yandex.direct.core.entity.timetarget.service;

import java.time.zone.ZoneRulesException;
import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.util.concurrent.UncheckedExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;
import ru.yandex.direct.core.entity.timetarget.repository.GeoTimezoneRepository;

import static ru.yandex.direct.libs.timetarget.TimeTargetUtils.DEFAULT_TIMEZONE;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;

/**
 * Сервис для работы с данными из таблицы {@link ru.yandex.direct.dbschema.ppcdict.tables.GeoTimezones}
 */
@Service
@ParametersAreNonnullByDefault
public class GeoTimezoneMappingService {
    private static final Logger logger = LoggerFactory.getLogger(GeoTimezoneMappingService.class);
    @SuppressWarnings("WeakerAccess")
    public static final GeoTimezone DEFAULT_GEO_TIMEZONE_RUS =
            new GeoTimezone()
                    .withRegionId(RUSSIA_REGION_ID)
                    .withTimezoneId(DEFAULT_TIMEZONE)
                    .withTimezone(MSK);

    private final Cache<Long, GeoTimezone> timezoneToRegion;

    private final GeoTimezoneRepository geoTimezoneRepository;

    @Autowired
    public GeoTimezoneMappingService(GeoTimezoneRepository geoTimezoneRepository) {
        this.geoTimezoneRepository = geoTimezoneRepository;

        timezoneToRegion =
                CacheBuilder.newBuilder()
                        .expireAfterWrite(10, TimeUnit.MINUTES)
                        .build();
    }

    /**
     * Позволяет по {@code timezoneId} (из, например, поля {@code CAMPAIGNS.TIMEZONE_ID}) получить ID региона,
     * который используется при определении праздничных дней в целевом регионе.
     */
    @Nonnull
    public GeoTimezone getRegionIdByTimezoneId(@Nullable Long timezoneId) {
        if (timezoneId == null || timezoneId == 0) {
            return DEFAULT_GEO_TIMEZONE_RUS;
        }

        try {
            return timezoneToRegion.get(timezoneId, () -> {
                Iterator<GeoTimezone> geoTimezones = geoTimezoneRepository.getGeoTimezonesByTimezoneIds(
                        Collections.singleton(timezoneId)).iterator();
                if (geoTimezones.hasNext()) {
                    return geoTimezones.next();
                } else {
                    logger.warn("Unknown timezoneId: {}", timezoneId);
                    return DEFAULT_GEO_TIMEZONE_RUS;
                }
            });
        } catch (UncheckedExecutionException | ExecutionException e) {
            if (e.getCause() instanceof ZoneRulesException) {
                logger.error("Error during getting regionId by timezoneId {}", timezoneId);
                return DEFAULT_GEO_TIMEZONE_RUS;
            }
            throw new RuntimeException("Error during getting regionId by timezoneId", e.getCause());
        }
    }
}
