package ru.yandex.direct.core.entity.timetarget.service;

import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.timetarget.model.HolidayItem;
import ru.yandex.direct.core.entity.timetarget.repository.ProductionCalendarRepository;
import ru.yandex.direct.libs.timetarget.ProductionCalendar;
import ru.yandex.direct.libs.timetarget.WeekdayType;
import ru.yandex.direct.solomon.SolomonUtils;

import static java.util.Collections.emptyList;

/**
 * Предоставляет методы для получения информации о производственном календаре ({@link ProductionCalendar}).
 */
@Service
@ParametersAreNonnullByDefault
public class ProductionCalendarProviderService {
    private static final Logger logger = LoggerFactory.getLogger(ProductionCalendarProviderService.class);

    private final ProductionCalendarRepository productionCalendarRepository;

    private final Cache<Integer, Map<Long, SingleRegionProductionCalendar>> holidaysByYears;

    @Autowired
    public ProductionCalendarProviderService(ProductionCalendarRepository productionCalendarRepository) {
        this.productionCalendarRepository = productionCalendarRepository;
        holidaysByYears =
                CacheBuilder.newBuilder()
                        .expireAfterWrite(1, TimeUnit.DAYS)
                        .build();

        SolomonUtils.registerGuavaCachesStats(
                "production_calendar_cache",
                Collections.singletonList(holidaysByYears)
        );
    }

    /**
     * @return Экземпляр {@link ProductionCalendar} с информацией о производственном календаре для выбранного региона
     * на указанный год.
     */
    @Nonnull
    public ProductionCalendar getProductionCalendar(Integer year, Long regionId) {
        try {
            Map<Long, SingleRegionProductionCalendar> holidayItemMap = holidaysByYears.get(year, () -> {
                        List<HolidayItem> holidaysByYear = productionCalendarRepository.getHolidaysByYear(year);
                        return StreamEx.of(holidaysByYear)
                                .mapToEntry(HolidayItem::getRegionId)
                                .invert()
                                .sorted(Map.Entry.comparingByKey())
                                .collapseKeys()
                                .mapValues(SingleRegionProductionCalendar::new)
                                .toMap();
                    }
            );

            return holidayItemMap.getOrDefault(regionId, new SingleRegionProductionCalendar(emptyList()));
        } catch (ExecutionException ex) {
            // ошибка при заполнении кэша
            logger.error("Error on getting holidays for year {}", year, ex);
            throw new RuntimeException("Error on getting holidays for year " + year, ex.getCause());
        }
    }

    public List<HolidayItem> getAllHolidays() {
        return productionCalendarRepository.getAllHolidays();
    }

    public int updateHolidays(Collection<HolidayItem> holidayItems) {
        return productionCalendarRepository.updateHolidays(holidayItems);
    }

    public int deleteHolidays(Collection<HolidayItem> holidayItems) {
        return productionCalendarRepository.deleteHolidays(holidayItems);
    }

    /**
     * Производственный календарь для отдельного региона
     */
    private static class SingleRegionProductionCalendar implements ProductionCalendar {

        private final Map<LocalDate, WeekdayType> singleRegionHolidayMap;

        private SingleRegionProductionCalendar(List<HolidayItem> holidays) {
            singleRegionHolidayMap = StreamEx.of(holidays)
                    .mapToEntry(HolidayItem::getDate, HolidayItem::getType)
                    .mapValues(ProductionCalendarProviderService::mapDbValueToWeekendType)
                    .toMap();
        }

        @Override
        public WeekdayType getWeekdayType(LocalDate date) {
            WeekdayType weekdayType = singleRegionHolidayMap.get(date);
            if (weekdayType != null) {
                return weekdayType;
            }
            return WeekdayType.getById(date.getDayOfWeek().getValue());
        }

    }

    private static WeekdayType mapDbValueToWeekendType(HolidayItem.Type type) {
        switch (type) {
            case HOLIDAY:
                return WeekdayType.HOLIDAY;
            case WORKDAY:
                return WeekdayType.WORKING_WEEKEND;
            default:
                throw new IllegalStateException("Unexpected value for enum WeekdayType: " + type);
        }
    }
}
