package ru.yandex.direct.core.entity.trustedredirects.repository;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.trustedredirects.model.Opts;
import ru.yandex.direct.core.entity.trustedredirects.model.RedirectType;
import ru.yandex.direct.core.entity.trustedredirects.model.TrustedRedirects;
import ru.yandex.direct.dbschema.ppcdict.enums.TrustedRedirectsRedirectType;
import ru.yandex.direct.dbschema.ppcdict.tables.records.TrustedRedirectsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.convertibleEnumSet;
import static ru.yandex.direct.dbschema.ppcdict.tables.TrustedRedirects.TRUSTED_REDIRECTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Репозиторий для работы с таблицей, содержащей данные о доверенных счётчиках переходов и сокращателей ссылок ppcdict.trusted_redirects
 */
@Repository
@ParametersAreNonnullByDefault
public class TrustedRedirectsRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<TrustedRedirects> trustedRedirectsJooqMapper;
    private final Set<Field<?>> fieldsToRead;

    @Autowired
    public TrustedRedirectsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        trustedRedirectsJooqMapper = JooqMapperWithSupplierBuilder.builder(TrustedRedirects::new)
                .map(property(TrustedRedirects.DOMAIN, TRUSTED_REDIRECTS.DOMAIN))
                .map(convertibleProperty(TrustedRedirects.REDIRECT_TYPE, TRUSTED_REDIRECTS.REDIRECT_TYPE,
                        RedirectType::fromSource, RedirectType::toSource))
                .map(convertibleEnumSet(TrustedRedirects.OPTS, TRUSTED_REDIRECTS.OPTS, Opts.class))
                .build();
        fieldsToRead = trustedRedirectsJooqMapper.getFieldsToRead();
    }

    /**
     * Возвращает записи из таблицы ppcdict.trusted_redirects с типом редиректа counter
     */
    public List<TrustedRedirects> getCounterTrustedRedirects() {
        return getTrustedDomains(TrustedRedirectsRedirectType.counter);
    }

    /**
     * @return Список доверенных доменов трекинговых систем нужного типа
     */
    public List<TrustedRedirects> getTrustedDomains(TrustedRedirectsRedirectType trustedRedirectsRedirectType) {
        return dslContextProvider.ppcdict()
                .select(fieldsToRead)
                .from(TRUSTED_REDIRECTS)
                .where(TRUSTED_REDIRECTS.REDIRECT_TYPE.eq(trustedRedirectsRedirectType))
                .fetch().map(trustedRedirectsJooqMapper::fromDb);
    }

    public boolean deleteDomain(TrustedRedirects trustedRedirect) {
        return dslContextProvider.ppcdict()
                .deleteFrom(TRUSTED_REDIRECTS)
                .where(TRUSTED_REDIRECTS.REDIRECT_TYPE.eq(RedirectType.toSource(trustedRedirect.getRedirectType())))
                .and(TRUSTED_REDIRECTS.DOMAIN.eq(trustedRedirect.getDomain()))
                .execute() > 0;
    }

    /**
     * Добавить доверенный домен
     */
    public boolean addTrustedDomain(TrustedRedirects trustedRedirect) {
        InsertHelper<TrustedRedirectsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppcdict(), TRUSTED_REDIRECTS);
        insertHelper.add(trustedRedirectsJooqMapper, trustedRedirect);
        return insertHelper.onDuplicateKeyIgnore().execute() > 0;
    }
}
