package ru.yandex.direct.core.entity.trustedredirects.service;


import java.net.IDN;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.trustedredirects.model.Opts;
import ru.yandex.direct.core.entity.trustedredirects.model.TrustedRedirects;
import ru.yandex.direct.core.entity.trustedredirects.repository.TrustedRedirectsRepository;
import ru.yandex.direct.dbschema.ppcdict.enums.TrustedRedirectsRedirectType;


/**
 * Класс предоставляющий способ ленивой инициализации набора domains для трекинговых ссылок.
 * <p>
 * Лениво – поскольку не в каждой операции, которая инстанцирует у себя этот объект,
 * могут быть нужны домены для валидации, т.е. чтобы лишний раз не дёргать ppcdict.
 */
class CounterDomainSet {

    private final TrustedRedirectsRepository trustedRedirectsRepository;
    private static final Logger logger = LoggerFactory.getLogger(CounterDomainSet.class);

    private static final int CRITICAL_MEMORY_DICT_SIZE = 1000;


    private Map<String, Set<Opts>> domainOpts;
    private final TrustedRedirectsRedirectType trustedRedirectsRedirectType;

    public CounterDomainSet(TrustedRedirectsRepository trustedRedirectsRepository, TrustedRedirectsRedirectType trustedRedirectsRedirectType) {
        this.trustedRedirectsRepository = trustedRedirectsRepository;
        this.trustedRedirectsRedirectType = trustedRedirectsRedirectType;
    }

    /**
     * Получить опции, установленные для данного домена.
     *
     * @param domain трекинговый домен
     * @return Набор опций {@link Opts} для данного домена или {@code null},
     * если домен отсутствует в таблице доверенных.
     */
    @Nullable
    public Set<Opts> getOpts(String domain) {
        if (domainOpts == null) {
            List<TrustedRedirects> counterDomains = trustedRedirectsRepository.getTrustedDomains(
                    trustedRedirectsRedirectType);
            if (counterDomains.size() >= CRITICAL_MEMORY_DICT_SIZE) {
                logger.error("Too large domains collection in db, there is critical size of %d items."
                        + " Increase limit or change to cache realization (DIRECT-67452).", CRITICAL_MEMORY_DICT_SIZE);
            }
            domainOpts = StreamEx.of(counterDomains)
                    .mapToEntry(TrustedRedirects::getDomain, TrustedRedirects::getOpts)
                    .flatMapKeys(s -> Stream.of(IDN.toASCII(s), IDN.toUnicode(s)))
                    .distinctKeys()
                    .toMap();
        }
        return domainOpts.get(domain);
    }
}
