package ru.yandex.direct.core.entity.trustedredirects.service;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.trustedredirects.model.Opts;
import ru.yandex.direct.core.entity.trustedredirects.model.TrustedRedirects;
import ru.yandex.direct.core.entity.trustedredirects.repository.TrustedRedirectsRepository;
import ru.yandex.direct.dbschema.ppcdict.enums.TrustedRedirectsRedirectType;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsUrlUtils.extractHostFromHrefWithoutWww;
import static ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsUrlUtils.extractProtocolFromHref;
import static ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsUrlUtils.extractSecondLevelDomainFromHost;
import static ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsUrlUtils.stripSubDomain;
import static ru.yandex.direct.core.validation.constraints.Constraints.validDomain;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

/**
 * Сервис для работы с таблицей, содержащей данные о доверенных счётчиках переходов и сокращателей ссылок ppcdict
 * .trusted_redirects
 */
@Service
@ParametersAreNonnullByDefault
public class TrustedRedirectsService {
    private final TrustedRedirectsRepository trustedRedirectsRepository;


    /**
     * Таблица доверенных счётчиков и сокращателей ссылок в базе содержит небольшое количество записей и очень редко
     * расширяется.
     * Для избегания ненужных обращений к постоянным данным, таблица загружается в память
     * и обрабатыается с учётом различных вариантов записи домена.
     * <p>
     * Записи загружаются при первом обращении к данным.
     * Домены обрабатываются и хранятся в памяти в двух возможных вариантах записи (punycode, unicode).
     */
    private CounterDomainSet counterDomainSet;
    private CounterDomainSet mobileAppCounterDomainSet;
    private CounterDomainSet mobileAppImpressionCounterDomainSet;


    private static final int MAX_DOMAIN_URL_SIZE = 100;


    public enum Result {
        /**
         * Доверенный URL.
         */
        TRUSTED,
        /**
         * Домен из URL не найден в списке доверенных.
         */
        NOT_TRUSTED,
        /**
         * Домен доверенный, но можно использовать только протокол HTTPS,
         * а в переданном URL неподходящий протокол.
         */
        HTTPS_REQUIRED
    }

    @Autowired
    public TrustedRedirectsService(TrustedRedirectsRepository trustedRedirectsRepository) {
        this.trustedRedirectsRepository = trustedRedirectsRepository;
        initDomainSets();
    }

    public void invalidateCache() {
        initDomainSets();
    }

    private void initDomainSets() {
        counterDomainSet = new CounterDomainSet(trustedRedirectsRepository,
                TrustedRedirectsRedirectType.counter);
        mobileAppCounterDomainSet = new CounterDomainSet(trustedRedirectsRepository,
                TrustedRedirectsRedirectType.mobile_app_counter);
        mobileAppImpressionCounterDomainSet = new CounterDomainSet(trustedRedirectsRepository,
                TrustedRedirectsRedirectType.mobile_app_impression_counter);
    }

    /**
     * Возвращает все изветные сокращатели ссылок-счетчики
     */
    public List<TrustedRedirects> getCounterTrustedRedirects() {
        return trustedRedirectsRepository.getCounterTrustedRedirects();
    }

    public boolean deleteDomain(TrustedRedirects trustedRedirect) {
        return trustedRedirectsRepository.deleteDomain(trustedRedirect);
    }

    public boolean addTrustedDomain(TrustedRedirects trustedRedirect) {
        // prepare domain from url
        trustedRedirect.setDomain(TrustedRedirectsUrlUtils.extractHostFromHrefOrNull(trustedRedirect.getDomain()));

        if (validate(trustedRedirect).hasAnyErrors()) {
            return false;
        }

        return trustedRedirectsRepository.addTrustedDomain(trustedRedirect);
    }

    private static ValidationResult<TrustedRedirects, Defect> validate(TrustedRedirects trustedRedirects) {
        ModelItemValidationBuilder<TrustedRedirects> vb = ModelItemValidationBuilder.of(trustedRedirects);
        vb.item(TrustedRedirects.DOMAIN)
                .check(maxStringLength(MAX_DOMAIN_URL_SIZE))
                .check(notBlank(), When.isValid())
                .check(validDomain(), When.isValid());
        return vb.getResult();
    }


    /**
     * ходим в базу максимум один раз. Доверенные домены кешируются
     */
    public boolean isTrusted(String href) {
        return isDomainTrusted(extractHostFromHrefWithoutWww(href));
    }

    /**
     * содержит ли справочник доменов переданную ссылку в виде домена или домена второго уровня
     * тип записи ссылки не фиксирован, словарь содержит оба варианта написания (punycode, unicode)
     */
    public boolean isDomainTrusted(String domain) {
        String secondLevelDomain = extractSecondLevelDomainFromHost(domain);
        return containsIfNotEmpty(domain) || containsIfNotEmpty(secondLevelDomain);
    }

    /**
     * Проверка вхождения в словарь значения, если значение не пустое
     */
    private boolean containsIfNotEmpty(String value) {
        return !value.isEmpty() && counterDomainSet.getOpts(value) != null;
    }


    /**
     * Проверить трекинговую ссылку. Результат проверки зависит от протокола URL.
     * Если протокол https, то домен будет считаться доверенным, если найдётся в таблице.
     * В случае протокола http – если будет найден, и в таблице на нём нет ограничения "https_only".
     * <p>
     * Проверяется с учётом настройки "разрешать субдомены", {@link Opts#allow_wildcard}.
     * <p>
     * Ходим в базу максимум один раз. Доверенные домены кешируются.
     *
     * @param trackingHref трекинговая ссылка установок мобильного приложения
     * @return {@link TrustedRedirectsService.Result}.
     */
    public Result checkTrackingHref(String trackingHref) {
        return checkUrl(trackingHref, mobileAppCounterDomainSet);
    }

    public Result checkImpressionUrl(String impressionUrl) {
        return checkUrl(impressionUrl, mobileAppImpressionCounterDomainSet);
    }

    private Result checkUrl(String url, CounterDomainSet domainSet) {
        String domain = extractHostFromHrefWithoutWww(url);
        boolean https = isHttps(extractProtocolFromHref(url));
        Result domainResult = check(domain, domainSet, https);
        if (domainResult == Result.NOT_TRUSTED) {
            return check(extractSecondLevelDomainFromHost(domain), domainSet, https);
        } else {
            return domainResult;
        }
    }

    private Result check(String domain, CounterDomainSet domainSet, boolean https) {
        Set<Opts> opts = domainSet.getOpts(domain);
        if (opts != null) {
            return opts.contains(Opts.https_only) && !https ? Result.HTTPS_REQUIRED : Result.TRUSTED;
        }
        opts = domainSet.getOpts(stripSubDomain(domain));
        if (opts != null && opts.contains(Opts.allow_wildcard)) {
            return opts.contains(Opts.https_only) && !https ? Result.HTTPS_REQUIRED : Result.TRUSTED;
        }
        return Result.NOT_TRUSTED;
    }

    private static boolean isHttps(String protocol) {
        return "https".equalsIgnoreCase(protocol);
    }
}
