package ru.yandex.direct.core.entity.trustedredirects.service;

import java.net.MalformedURLException;
import java.net.URL;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ParametersAreNonnullByDefault
class TrustedRedirectsUrlUtils {

    private static final Logger logger = LoggerFactory.getLogger(TrustedRedirectsUrlUtils.class);

    private TrustedRedirectsUrlUtils() {
    }

    /**
     * Ссылка должна быть предварительно провалидирована.
     * Возвращает домен без префикса www, если он есть
     * В случае неправильной ссылки возвращается пустая строка
     */
    static String extractHostFromHrefWithoutWww(String href) {
        try {
            String host = new URL(href).getHost();
            return host.startsWith("www.") ? host.substring(4) : host;
        } catch (MalformedURLException e) {
            // something wrong with current href, skip it
            logInvalidUrlError(href, e);
            return "";
        }
    }


    static String extractProtocolFromHref(String href) {
        try {
            return new URL(href).getProtocol();
        } catch (MalformedURLException e) {
            logInvalidUrlError(href, e);
            return "";
        }
    }


    /**
     * Возвращает домен второго уровня от переданного хоста
     * или пустая строка, если выделить домен не удаётся
     */
    static String extractSecondLevelDomainFromHost(String host) {
        String[] tokens = host.split("\\.");
        int length = tokens.length;
        return length >= 2 ? tokens[length - 2] + "." + tokens[length - 1] : "";
    }

    static String stripSubDomain(String domain) {
        String secondLevel = extractSecondLevelDomainFromHost(domain);
        if (secondLevel.equals(domain)) {
            return domain;
        }
        int dotIdx = domain.indexOf('.');
        return dotIdx == -1 ? domain : domain.substring(dotIdx + 1);
    }

    @Nullable
    static String extractHostFromHrefOrNull(@Nullable String href) {
        if (href == null) {
            return null;
        }
        try {
            return new URL(href).getHost();
        } catch (MalformedURLException e) {
            return href;
        }
    }

    private static void logInvalidUrlError(String href, Exception ex) {
        logger.error("URI build or parse error for value %s. Href must be already validated. Skip error.", href, ex);
    }

}
