package ru.yandex.direct.core.entity.turbolanding.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Result;
import org.jooq.SelectOnConditionStep;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.banner.model.old.OldBanner;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.turbolanding.model.OldBannerTurboLandingStatusModerate;
import ru.yandex.direct.core.entity.moderation.service.receiving.BaseModerationReceivingService.ModeratedObjectKeyWithVersion;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingMetrikaCounter;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingMetrikaCounterContainer;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingMetrikaGoal;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingMetrikaGoalContainer;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingWithCountersAndGoals;
import ru.yandex.direct.dbschema.ppc.enums.BannerTurbolandingsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.TurbolandingsStatusmoderateforcpa;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerTurbolandingsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.TurbolandingMetrikaCountersRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.TurbolandingMetrikaGoalsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.TurbolandingsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.emptyList;
import static org.jooq.impl.DSL.falseCondition;
import static ru.yandex.direct.core.entity.moderation.service.receiving.BaseModerationReceivingService.ModeratedObjectKeyWithVersion.ANY_VERSION;
import static ru.yandex.direct.core.entity.turbolanding.repository.mapper.TurboLandingMapperProvider.getTurboLandingMapper;
import static ru.yandex.direct.core.entity.turbolanding.repository.mapper.TurboLandingMetrikaCounterMapperProvider.getTurboLandingMetrikaCounterMapper;
import static ru.yandex.direct.core.entity.turbolanding.repository.mapper.TurboLandingMetrikaCounterMapperProvider.getTurboLandingMetrikaGoalMapper;
import static ru.yandex.direct.dbschema.ppc.tables.BannerTurboModerationVersions.BANNER_TURBO_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerTurbolandings.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.SitelinksLinks.SITELINKS_LINKS;
import static ru.yandex.direct.dbschema.ppc.tables.SitelinksSetToLink.SITELINKS_SET_TO_LINK;
import static ru.yandex.direct.dbschema.ppc.tables.TurbolandingMetrikaCounters.TURBOLANDING_METRIKA_COUNTERS;
import static ru.yandex.direct.dbschema.ppc.tables.TurbolandingMetrikaGoals.TURBOLANDING_METRIKA_GOALS;
import static ru.yandex.direct.dbschema.ppc.tables.Turbolandings.TURBOLANDINGS;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class TurboLandingRepository {
    public static final String GOAL_ROLE_COMBINED = "combined";

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final JooqMapperWithSupplier<TurboLandingMetrikaCounterContainer>
            turboLandingMetrikaCounterMapper;
    private final JooqMapperWithSupplier<TurboLandingMetrikaGoalContainer>
            turboLandingMetrikaGoalMapper;

    @Autowired
    public TurboLandingRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        turboLandingMetrikaCounterMapper = getTurboLandingMetrikaCounterMapper();
        turboLandingMetrikaGoalMapper = getTurboLandingMetrikaGoalMapper();
    }

    /**
     * По заданному clientId и списку turbolandingId возвращает список турболендингов, принадлежащих указанному клиенту
     *
     * @param shard           номер шарда
     * @param clientId        идентификатор клиента
     * @param turbolandingIds список идентификаторов турболендингов
     * @param limitOffset     ограничение на количество записей в результате и смещение
     */
    public List<TurboLanding> getClientTurboLandingsbyId(int shard, ClientId clientId,
                                                         Collection<Long> turbolandingIds, LimitOffset limitOffset) {
        return dslContextProvider.ppc(shard)
                .select(getTurboLandingMapper().getFieldsToRead())
                .from(TURBOLANDINGS)
                .where(TURBOLANDINGS.CLIENT_ID.eq(clientId.asLong()))
                .and(TURBOLANDINGS.TL_ID.in(turbolandingIds))
                .offset(limitOffset.offset())
                .limit(limitOffset.limit())
                .fetch()
                .map(t -> getTurboLandingMapper().fromDb(t));
    }


    /**
     * По заданному clientId и списку turbolandingId возвращает список турболендингов, принадлежащих указанному клиенту
     *
     * @param shard           номер шарда
     * @param clientId        идентификатор клиента
     * @param turbolandingIds список идентификаторов турболендингов
     */
    public List<TurboLanding> getClientTurboLandingsbyId(int shard, ClientId clientId,
                                                         Collection<Long> turbolandingIds) {
        return getClientTurboLandingsbyId(shard, clientId, turbolandingIds, LimitOffset.maxLimited());
    }

    /**
     * По заданному clientId возвращает список турболендингов, принадлежащих указанному клиенту, всех или по списку id
     *
     * @param shard           номер шарда
     * @param clientId        идентификатор клиента
     * @param turbolandingIds список идентификаторов турболендингов
     * @param limitOffset     ограничение на количество записей в результате и смещение
     */
    public List<TurboLanding> getClientTurbolandings(int shard, ClientId clientId,
                                                     @Nullable Collection<Long> turbolandingIds,
                                                     LimitOffset limitOffset) {
        if (CollectionUtils.isNotEmpty(turbolandingIds)) {
            return getClientTurboLandingsbyId(shard, clientId, turbolandingIds, limitOffset);
        }

        return dslContextProvider.ppc(shard)
                .select(getTurboLandingMapper().getFieldsToRead())
                .from(TURBOLANDINGS)
                .where(TURBOLANDINGS.CLIENT_ID.eq(clientId.asLong()))
                .offset(limitOffset.offset())
                .limit(limitOffset.limit())
                .fetch()
                .map(t -> getTurboLandingMapper().fromDb(t));
    }


    /**
     * По заданному clientId возвращает список турболендингов, принадлежащих указанному клиенту, всех или по списку id
     *
     * @param shard           номер шарда
     * @param clientId        идентификатор клиента
     * @param turbolandingIds список идентификаторов турболендингов
     */
    public List<TurboLanding> getClientTurbolandings(int shard, ClientId clientId,
                                                     @Nullable Collection<Long> turbolandingIds) {
        return getClientTurbolandings(shard, clientId, turbolandingIds, LimitOffset.maxLimited());
    }


    /**
     * По заданному clientId и списку turbolandingId возвращает идентификаторы турболендингов,
     * принадлежащих указанному клиенту
     *
     * @param shard           номер шарда
     * @param clientId        идентификатор клиента
     * @param turbolandingIds список идентификаторов турболендингов
     */
    public Set<Long> getClientTurboLandingIdsById(int shard, ClientId clientId, Collection<Long> turbolandingIds) {
        return dslContextProvider.ppc(shard)
                .select(TURBOLANDINGS.TL_ID)
                .from(TURBOLANDINGS)
                .where(TURBOLANDINGS.CLIENT_ID.eq(clientId.asLong()))
                .and(TURBOLANDINGS.TL_ID.in(turbolandingIds))
                .fetchSet(TURBOLANDINGS.TL_ID);
    }

    public List<TurboLanding> getCpaTurbolandingsReadyForModeration(int shard, LimitOffset limitOffset) {
        return dslContextProvider.ppc(shard)
                .select(getTurboLandingMapper().getFieldsToRead())
                .from(TURBOLANDINGS)
                .where(TURBOLANDINGS.IS_CPA_MODERATION_REQUIRED.eq(1L)
                        .and(TURBOLANDINGS.STATUS_MODERATE_FOR_CPA.in(TurbolandingsStatusmoderateforcpa.Ready,
                                TurbolandingsStatusmoderateforcpa.Sending)))
                .offset(limitOffset.offset())
                .limit(limitOffset.limit())
                .fetch()
                .map(t -> getTurboLandingMapper().fromDb(t));
    }

    public List<TurboLanding> selectAndLockTurbolanding(int shard, LimitOffset limitOffset) {
        return dslContextProvider.ppc(shard)
                .select(getTurboLandingMapper().getFieldsToRead())
                .from(TURBOLANDINGS)
                .where(TURBOLANDINGS.IS_CPA_MODERATION_REQUIRED.eq(1L)
                        .and(TURBOLANDINGS.STATUS_MODERATE_FOR_CPA.eq(TurbolandingsStatusmoderateforcpa.Ready)))
                .offset(limitOffset.offset())
                .limit(limitOffset.limit())
                .forUpdate()
                .fetch()
                .map(t -> getTurboLandingMapper().fromDb(t));
    }

    public int updateTurbolandingsStatusModerateForCpa(int shard, Collection<Long> turbolandingIds,
                                                       TurbolandingsStatusmoderateforcpa statusModerateForCpaFrom,
                                                       TurbolandingsStatusmoderateforcpa statusModerateForCpaTo) {

        return dslContextProvider.ppc(shard)
                .update(TURBOLANDINGS)
                .set(TURBOLANDINGS.STATUS_MODERATE_FOR_CPA, statusModerateForCpaTo)
                .where(TURBOLANDINGS.TL_ID.in(turbolandingIds)
                        .and(TURBOLANDINGS.STATUS_MODERATE_FOR_CPA.eq(statusModerateForCpaFrom)))
                .execute();
    }


    public List<Long> selectByIdsAndExportVersionsWithLock(Configuration configuration,
                                                           Collection<ModeratedObjectKeyWithVersion<Long>> bidAndVersions) {
        if (bidAndVersions.isEmpty()) {
            return List.of();
        }

        SelectOnConditionStep<Record1<Long>> select = DSL.using(configuration)
                .select(BANNER_TURBOLANDINGS.BID)
                .from(BANNER_TURBOLANDINGS)
                .join(BANNER_TURBO_MODERATION_VERSIONS)
                .on(BANNER_TURBOLANDINGS.BID.eq(BANNER_TURBO_MODERATION_VERSIONS.BID));

        Condition condition = DSL.falseCondition();

        for (ModeratedObjectKeyWithVersion<Long> response : bidAndVersions) {
            Condition subCondition = BANNER_TURBOLANDINGS.BID.eq(response.getKey());

            if (response.getVersionId() != ANY_VERSION) {
                subCondition = subCondition.and(BANNER_TURBO_MODERATION_VERSIONS.VERSION.eq(response.getVersionId()));
            }

            condition = condition.or(subCondition);
        }

        return select.where(condition)
                .forUpdate()
                .fetch(el -> el.get(BANNER_TURBOLANDINGS.BID));
    }

    /**
     * Переотправить на модерацию турбоссылки которые были отправлены (в статусе Sent)
     */
    public void resendTurboLandingToModeration(
            Configuration configuration, List<Long> bids) {
        if (bids.isEmpty()) {
            return;
        }

        configuration.dsl()
                .update(BANNER_TURBOLANDINGS)
                .set(BANNER_TURBOLANDINGS.STATUS_MODERATE, BannerTurbolandingsStatusmoderate.Ready)
                .where(BANNER_TURBOLANDINGS.BID.in(bids),
                        BANNER_TURBOLANDINGS.STATUS_MODERATE.eq(BannerTurbolandingsStatusmoderate.Sent)
                )
                .execute();
    }


    /**
     * Добавляет в таблицу BANNER_TURBOLANDINGS новые записи или обновляет существующие.
     * Статус модерации турболендинга должен быть выставлен до вызова этого метода.
     *
     * @param banners баннеры, которые нужно добавить
     * @param context контекст
     * @param <T>     тип баннеров, поддерживающих турболендинги
     */
    public <T extends OldBannerWithTurboLanding> void addBannerToBannerTurbolandingsTableOrUpdate(Collection<T> banners,
                                                                                                  DSLContext context) {
        InsertHelper<BannerTurbolandingsRecord> insertHelper =
                new InsertHelper<>(context, BANNER_TURBOLANDINGS);

        Predicate<OldBannerWithTurboLanding> hasTurbolanding = banner -> banner.getTurboLandingId() != null;

        banners.stream().filter(hasTurbolanding)
                .forEach(
                        b -> {
                            if (b.getTurboLandingStatusModerate() == null) {
                                throw new IllegalArgumentException(
                                        "Null turbolandingStatusModerate in banner with id " + b.getId());
                            }
                            insertHelper.set(BANNER_TURBOLANDINGS.TL_ID, b.getTurboLandingId());
                            insertHelper.set(BANNER_TURBOLANDINGS.BID, b.getId());
                            insertHelper.set(BANNER_TURBOLANDINGS.CID, b.getCampaignId());
                            insertHelper.set(BANNER_TURBOLANDINGS.STATUS_MODERATE,
                                    OldBannerTurboLandingStatusModerate.toSource(b.getTurboLandingStatusModerate()));
                            insertHelper.newRecord();
                        }
                );
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(BANNER_TURBOLANDINGS.TL_ID, MySQLDSL.values(BANNER_TURBOLANDINGS.TL_ID))
                    .set(BANNER_TURBOLANDINGS.STATUS_MODERATE, MySQLDSL.values(BANNER_TURBOLANDINGS.STATUS_MODERATE));
            insertHelper.execute();
        }
    }

    /**
     * Удаляет запись из таблицы BANNER_TURBOLANDINGS.
     *
     * @param shard   номер шарда
     * @param banners баннеры, которые нужно удалить
     * @param <T>     тип баннеров, поддерживающих турболендинги
     */
    public <T extends OldBannerWithTurboLanding> void deleteBannerFromBannerTurboLandingsTable(int shard,
                                                                                               Collection<T> banners) {
        deleteBannerFromBannerTurboLandingsTable(banners, dslContextProvider.ppc(shard));
    }

    /**
     * Удаляет запись из таблицы BANNER_TURBOLANDINGS.
     *
     * @param banners баннеры, которые нужно удалить
     * @param context контекст
     * @param <T>     тип баннеров, поддерживающих турболендинги
     */
    public <T extends OldBannerWithTurboLanding> void deleteBannerFromBannerTurboLandingsTable(Collection<T> banners,
                                                                                               DSLContext context) {
        context.delete(BANNER_TURBOLANDINGS)
                .where(BANNER_TURBOLANDINGS.BID.in(
                        mapList(banners, OldBanner::getId)))
                .execute();
    }

    /**
     * По заданному списку bannerIds возвращает список TurboLanding,
     * для баннеров с турболендингами и сайтлинков с турболендингами которые связаны с банерами
     *
     * @param context   контекст
     * @param bannerIds список bid, для которых нужно получить турболендинги
     */
    public Map<Long, TurboLanding> getTurbolandingsLinkedWithBanners(DSLContext context, List<Long> bannerIds) {
        Result<Record1<Long>> bannerTurbolandingIds = context
                .select(BANNER_TURBOLANDINGS.TL_ID)
                .from(BANNER_TURBOLANDINGS)
                .where(BANNER_TURBOLANDINGS.BID.in(bannerIds))
                .fetch();

        Result<Record1<Long>> sitelinkTurboLandingIds = context
                .select(SITELINKS_LINKS.TL_ID)
                .from(BANNERS)
                .join(SITELINKS_SET_TO_LINK).on(BANNERS.SITELINKS_SET_ID.eq(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID))
                .join(SITELINKS_LINKS).on(SITELINKS_LINKS.SL_ID.eq(SITELINKS_SET_TO_LINK.SL_ID))
                .where(BANNERS.BID.in(bannerIds))
                .and(SITELINKS_LINKS.SL_ID.greaterThan(0L))
                .fetch();

        bannerTurbolandingIds.addAll(sitelinkTurboLandingIds);
        Set<Long> turbolandingIds = bannerTurbolandingIds.stream().map(Record1::value1).collect(Collectors.toSet());

        List<Field<?>> fields = new ArrayList<>(getTurboLandingMapper().getFieldsToRead());
        fields.add(BANNER_TURBOLANDINGS.BID);
        Result<Record> turbolandings = context
                .select(fields)
                .from(TURBOLANDINGS)
                .join(BANNER_TURBOLANDINGS)
                .on(TURBOLANDINGS.TL_ID.eq(BANNER_TURBOLANDINGS.TL_ID))
                .where(TURBOLANDINGS.TL_ID.in(turbolandingIds))
                .fetch();

        return StreamEx.of(turbolandings)
                .mapToEntry(r -> (r.get(BANNER_TURBOLANDINGS.BID)), getTurboLandingMapper()::fromDb).toMap();
    }

    public Map<Long, TurboLanding> getTurboLandings(int shard, Collection<Long> turbolandingIds) {
        var mapper = getTurboLandingMapper();
        return dslContextProvider.ppc(shard).select(mapper.getFieldsToRead())
                .from(TURBOLANDINGS)
                .where(TURBOLANDINGS.TL_ID.in(turbolandingIds))
                .fetchMap(r -> r.get(TURBOLANDINGS.TL_ID), mapper::fromDb);
    }

    /**
     * По заданному идентификатору турболендинга возврашает список идентификаторов не архивных баннеров,
     * связанных с ним через сайтлинки
     *
     * @param shard
     * @param clientId
     * @param turboLandingId
     * @return
     */
    public List<Long> getBannerIdsLinkedWithTurboLandingViaSitelink(int shard, ClientId clientId, long turboLandingId) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(BANNERS.BID)
                .from(SITELINKS_LINKS)
                .join(SITELINKS_SET_TO_LINK).on(SITELINKS_LINKS.SL_ID.eq(SITELINKS_SET_TO_LINK.SL_ID))
                .join(BANNERS).on(BANNERS.SITELINKS_SET_ID.eq(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(SITELINKS_LINKS.TL_ID.eq(turboLandingId))
                .and(BANNERS.STATUS_ARCH.ne(BannersStatusarch.Yes))
                .and(CAMPAIGNS.ARCHIVED.ne(CampaignsArchived.Yes))
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .fetch(Record1::value1);
    }

    /**
     * По заданному идентификатору турболендинга возвращает список идентификаторов не архивных баннеров,
     * связанных с ним через основную ссылку
     *
     * @param shard
     * @param clientId
     * @param turboLandingId
     * @return
     */
    public List<Long> getBannerIdsLinkedWithTurboLandingDirectly(int shard, ClientId clientId, long turboLandingId) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(BANNER_TURBOLANDINGS.BID)
                .from(BANNER_TURBOLANDINGS)
                .join(BANNERS).on(BANNERS.BID.eq(BANNER_TURBOLANDINGS.BID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNER_TURBOLANDINGS.TL_ID.eq(turboLandingId))
                .and(BANNERS.STATUS_ARCH.ne(BannersStatusarch.Yes))
                .and(CAMPAIGNS.ARCHIVED.ne(CampaignsArchived.Yes))
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .fetch(Record1::value1);
    }

    /**
     * Сохраняет переданные турболендинги в БД
     *
     * @param shard         номер шарда
     * @param turboLandings список сохраняемых турболендингов
     */
    public void add(int shard, List<TurboLanding> turboLandings) {
        if (turboLandings.size() == 0) {
            return;
        }

        InsertHelper<TurbolandingsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), TURBOLANDINGS);
        insertHelper.addAll(getTurboLandingMapper(), turboLandings);

        insertHelper.execute();
    }

    public void addOrUpdateTurboLandings(int shard,
                                         List<TurboLandingWithCountersAndGoals> turboLandingsWithCountersAndGoals) {
        if (turboLandingsWithCountersAndGoals.isEmpty()) {
            return;
        }

        Map<Long, Set<TurboLandingMetrikaCounter>> countersByTurboLandingId =
                listToMap(turboLandingsWithCountersAndGoals,
                        TurboLandingWithCountersAndGoals::getId, TurboLandingWithCountersAndGoals::getCounters);


        Map<Long, Set<TurboLandingMetrikaGoal>> goalsByTurboLandingId =
                listToMap(turboLandingsWithCountersAndGoals,
                        TurboLandingWithCountersAndGoals::getId, TurboLandingWithCountersAndGoals::getGoals);


        updateTurboLandingsTable(shard, turboLandingsWithCountersAndGoals);
        updateTurboLandingMetrikaCountersTable(shard, countersByTurboLandingId);
        updateTurboLandingMetrikaGoalsTable(shard, goalsByTurboLandingId);
    }


    private <T extends TurboLanding> void updateTurboLandingsTable(int shard, List<T> turboLandings) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        InsertHelper<TurbolandingsRecord> insertHelper = new InsertHelper<>(dslContext, TURBOLANDINGS)
                .addAll(getTurboLandingMapper(), turboLandings);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(TURBOLANDINGS.NAME, MySQLDSL.values(TURBOLANDINGS.NAME))
                    .set(TURBOLANDINGS.METRIKA_COUNTERS_JSON, MySQLDSL.values(TURBOLANDINGS.METRIKA_COUNTERS_JSON))
                    .set(TURBOLANDINGS.HREF, MySQLDSL.values(TURBOLANDINGS.HREF))
                    .set(TURBOLANDINGS.PREVIEW_HREF, MySQLDSL.values(TURBOLANDINGS.PREVIEW_HREF))
                    .set(TURBOLANDINGS.TURBO_SITE_HREF, MySQLDSL.values(TURBOLANDINGS.TURBO_SITE_HREF))
                    .set(TURBOLANDINGS.STATUS_POST_MODERATE, MySQLDSL.values(TURBOLANDINGS.STATUS_POST_MODERATE))
                    .set(TURBOLANDINGS.LAST_MODERATED_VERSION, MySQLDSL.values(TURBOLANDINGS.LAST_MODERATED_VERSION))
                    .set(TURBOLANDINGS.VERSION, MySQLDSL.values(TURBOLANDINGS.VERSION))
                    .set(TURBOLANDINGS.STATUS_MODERATE_FOR_CPA, MySQLDSL.values(TURBOLANDINGS.STATUS_MODERATE_FOR_CPA))
                    .set(TURBOLANDINGS.IS_CHANGED, MySQLDSL.values(TURBOLANDINGS.IS_CHANGED))
                    .set(TURBOLANDINGS.PRESET, MySQLDSL.values(TURBOLANDINGS.PRESET));
        }
        insertHelper.executeIfRecordsAdded();
    }

    private void updateTurboLandingMetrikaCountersTable(
            int shard, Map<Long, Set<TurboLandingMetrikaCounter>> countersByTurboLandingId) {
        DSLContext dslContext = dslContextProvider.ppc(shard);

        //удаление
        Condition deleteCondition = falseCondition();
        for (Long turboLandingId : countersByTurboLandingId.keySet()) {
            List<Long> counterIds = mapList(countersByTurboLandingId.get(turboLandingId),
                    TurboLandingMetrikaCounter::getId);

            Condition condition = TURBOLANDING_METRIKA_COUNTERS.TL_ID.eq(turboLandingId)
                    .and(TURBOLANDING_METRIKA_COUNTERS.COUNTER.notIn(counterIds));

            deleteCondition = deleteCondition.or(condition);
        }
        dslContext.deleteFrom(TURBOLANDING_METRIKA_COUNTERS)
                .where(TURBOLANDING_METRIKA_COUNTERS.TL_ID.in(countersByTurboLandingId.keySet()))
                .and(deleteCondition)
                .execute();

        //добавление
        List<TurboLandingMetrikaCounterContainer> counters = EntryStream.of(countersByTurboLandingId)
                .flatMap(
                        (entry) -> entry.getValue().stream()
                                .map(counter -> new TurboLandingMetrikaCounterContainer()
                                        .withTurbolandingId(entry.getKey())
                                        .withCounterId(counter.getId())
                                        .withIsUserCounter(counter.getIsUserCounter())
                                )).toList();

        InsertHelper<TurbolandingMetrikaCountersRecord> insertHelper =
                new InsertHelper<>(dslContext, TURBOLANDING_METRIKA_COUNTERS)
                        .addAll(turboLandingMetrikaCounterMapper, counters);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(TURBOLANDING_METRIKA_COUNTERS.IS_USER_COUNTER,
                            MySQLDSL.values(TURBOLANDING_METRIKA_COUNTERS.IS_USER_COUNTER));
        }
        insertHelper.executeIfRecordsAdded();
    }

    private void updateTurboLandingMetrikaGoalsTable(int shard,
                                                     Map<Long, Set<TurboLandingMetrikaGoal>> goalsByTurboLandingId) {
        DSLContext dslContext = dslContextProvider.ppc(shard);

        //удаление
        Condition deleteCondition = falseCondition();
        for (Long turboLandingId : goalsByTurboLandingId.keySet()) {
            List<Long> goalIds = mapList(goalsByTurboLandingId.get(turboLandingId),
                    TurboLandingMetrikaGoal::getId);

            Condition condition = TURBOLANDING_METRIKA_GOALS.TL_ID.eq(turboLandingId)
                    .and(TURBOLANDING_METRIKA_GOALS.GOAL_ID.notIn(goalIds));

            deleteCondition = deleteCondition.or(condition);
        }
        dslContext.deleteFrom(TURBOLANDING_METRIKA_GOALS)
                .where(TURBOLANDING_METRIKA_GOALS.TL_ID.in(goalsByTurboLandingId.keySet()))
                .and(deleteCondition)
                .execute();

        //добавление
        List<TurboLandingMetrikaGoalContainer> goals = EntryStream.of(goalsByTurboLandingId)
                .flatMap(entry -> entry.getValue().stream()
                        .map(goal -> new TurboLandingMetrikaGoalContainer()
                                .withTurbolandingId(entry.getKey())
                                .withGoalId(goal.getId())
                                .withIsConversionGoal(
                                        goal.getIsConversionGoal() == null ? false : goal.getIsConversionGoal())
                        )
                ).toList();

        InsertHelper<TurbolandingMetrikaGoalsRecord> insertHelper =
                new InsertHelper<>(dslContext, TURBOLANDING_METRIKA_GOALS)
                        .addAll(turboLandingMetrikaGoalMapper, goals);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(TURBOLANDING_METRIKA_GOALS.IS_CONVERSION_GOAL,
                            MySQLDSL.values(TURBOLANDING_METRIKA_GOALS.IS_CONVERSION_GOAL));
        }
        insertHelper.executeIfRecordsAdded();
    }

    public List<TurboLandingMetrikaCounterContainer> getTurboLandingMetrikaCounters(int shard,
                                                                                    Collection<Long> turboLandingIds) {
        if (turboLandingIds == null || turboLandingIds.isEmpty()) {
            return emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(turboLandingMetrikaCounterMapper.getFieldsToRead())
                .from(TURBOLANDING_METRIKA_COUNTERS)
                .where(TURBOLANDING_METRIKA_COUNTERS.TL_ID.in(turboLandingIds))
                .fetch(turboLandingMetrikaCounterMapper::fromDb);
    }

    public List<TurboLandingMetrikaGoalContainer> getTurboLandingMetrikaGoals(int shard,
                                                                              Collection<Long> turboLandingIds) {
        if (turboLandingIds.isEmpty()) {
            return emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(turboLandingMetrikaGoalMapper.getFieldsToRead())
                .from(TURBOLANDING_METRIKA_GOALS)
                .where(TURBOLANDING_METRIKA_GOALS.TL_ID.in(turboLandingIds))
                .fetch(turboLandingMetrikaGoalMapper::fromDb);
    }

    public Set<Long> getSystemTurboLandingMetrikaCounterIdsByClientId(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return getSystemTurboLandingMetrikaCounterIdsByClientId(shard, clientId);
    }

    public Set<Long> getSystemTurboLandingMetrikaCounterIdsByClientId(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(TURBOLANDING_METRIKA_COUNTERS.COUNTER)
                .from(TURBOLANDING_METRIKA_COUNTERS)
                .join(TURBOLANDINGS).on(TURBOLANDINGS.TL_ID.eq(TURBOLANDING_METRIKA_COUNTERS.TL_ID))
                .where(TURBOLANDINGS.CLIENT_ID.eq(clientId.asLong()))
                .and(TURBOLANDING_METRIKA_COUNTERS.IS_USER_COUNTER.eq(RepositoryUtils.FALSE))
                .fetchSet(TURBOLANDING_METRIKA_COUNTERS.COUNTER);
    }
}
