package ru.yandex.direct.core.entity.turbolanding.service;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.turbolanding.model.StatusPostModerate;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingMetrikaCounter;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingWithCountersAndGoals;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.core.entity.turbolanding.service.validation.TurboLandingValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.traceinterception.entity.traceinterception.exception.TraceInterceptionSemaphoreException;
import ru.yandex.direct.turbolandings.client.TurboLandingsClient;
import ru.yandex.direct.turbolandings.client.TurboLandingsClientException;
import ru.yandex.direct.turbolandings.client.model.DcTurboLanding;
import ru.yandex.direct.turbolandings.client.model.GetIdByUrlResponseItem;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@Service
public class TurboLandingService {

    private static final Pattern TURBO_LANDING_URL_PATTERN = Pattern.compile("^https?://([\\w\\-]+\\.)*" +
            "((yandex\\.ru/turbo)" +
            "|(turbo\\.site)" +
            "|(turbo-site-integration\\.common\\.yandex\\.net)" +
            "|(turbo-site-test\\.common\\.yandex\\.net)" +
            "|(turbo-site\\.dc-stage\\.yandex\\.net))",
            Pattern.UNICODE_CHARACTER_CLASS);
    private static final Logger logger = LoggerFactory.getLogger(TurboLandingService.class);

    private final ShardHelper shardHelper;
    private final TurboLandingRepository turboLandingRepository;
    private final UpdateCounterGrantsService updateCounterGrantsService;
    private final TurboLandingValidationService turboLandingValidationService;
    private final TurboLandingsClient turboLandingsClient;

    @Autowired
    public TurboLandingService(
            ShardHelper shardHelper,
            TurboLandingRepository turboLandingRepository,
            UpdateCounterGrantsService updateCounterGrantsService,
            TurboLandingValidationService turboLandingValidationService, TurboLandingsClient turboLandingsClient) {
        this.shardHelper = shardHelper;
        this.turboLandingRepository = turboLandingRepository;
        this.updateCounterGrantsService = updateCounterGrantsService;
        this.turboLandingValidationService = turboLandingValidationService;
        this.turboLandingsClient = turboLandingsClient;
    }

    /**
     * По заданному clientId и списку turbolandingId возвращает список турболендингов, принадлежащих указанному клиенту
     *
     * @param clientId        идентификатор клиента
     * @param turbolandingIds список идентификаторов турболендингов
     */
    public List<TurboLanding> getClientTurboLandingsById(
            ClientId clientId, List<Long> turbolandingIds, LimitOffset limitOffset) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return turboLandingRepository.getClientTurbolandings(shard, clientId, turbolandingIds, limitOffset);
    }

    /**
     * По заданному clientId и списку turbolandingId возвращает список идентификаторов турболендингов,
     * принадлежащих указанному клиенту. Если для всех или части turbolandingId у указанного клиента
     * не удалось найти турболендингов в БД - эти id будут просто исключены из обработки.
     *
     * @param clientId        идентификатор клиента
     * @param turbolandingIds список идентификаторов турболендингов
     * @return
     */
    public Set<Long> fetchTurboLandingIds(ClientId clientId, Collection<Long> turbolandingIds) {
        if (clientId == null) {
            return emptySet();
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return turboLandingRepository.getClientTurboLandingIdsById(shard, clientId, turbolandingIds);
    }

    public void saveTurboLandings(long operatorUid, ClientId clientId,
                                  List<TurboLandingWithCountersAndGoals> turboLandings) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<Long, TurboLanding> existingTurboLandingsByIds =
                StreamEx.of(turboLandingRepository.getClientTurbolandings(shard, clientId, null))
                        .toMap(TurboLanding::getId, Function.identity());

        fillModerationParameters(turboLandings, existingTurboLandingsByIds);

        addMetrikaGrantsForNewCounters(operatorUid, clientId, turboLandings);

        turboLandingRepository.addOrUpdateTurboLandings(shard, turboLandings);
    }

    private void fillModerationParameters(List<TurboLandingWithCountersAndGoals> turboLandings,
                                          Map<Long, TurboLanding> existingTurboLandingsByIds) {
        for (TurboLanding turboLanding : turboLandings) {
            if (turboLanding.getIsCpaModerationRequired() && turboLanding.getStatusModerateForCpa() == null) {
                turboLanding.setStatusModerateForCpa(
                        existingTurboLandingsByIds.get(turboLanding.getId()).getStatusModerateForCpa());
            }
            //Todo(pashkus) упростить логику обновления полей пост-модерации (DIRECT-117131)
            if (turboLanding.getVersion() > 0) {
                turboLanding.setIsChanged(true);
                if (!existingTurboLandingsByIds.containsKey(turboLanding.getId())) {
                    turboLanding.setStatusPostModerate(StatusPostModerate.YES);
                    turboLanding.setLastModeratedVersion(0L);
                }
            } else {
                // Конструктор не поддерживает версионность, отправлять на модерацию не нужно
                turboLanding.setIsChanged(false);
                turboLanding.setStatusPostModerate(StatusPostModerate.YES);
                turboLanding.setLastModeratedVersion(0L);
            }
        }
    }

    /**
     *
     */
    public MassResult<Long> saveModeratableTurbolanding(long operatorUid, ClientId clientId,
                                                        List<TurboLandingWithCountersAndGoals> turboLandings) {

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        List<Long> turbolandingIds = mapList(turboLandings, TurboLanding::getId);

        Map<Long, TurboLanding> existingTurboLandingsByIds =
                StreamEx.of(turboLandingRepository.getClientTurbolandings(shard, clientId, null))
                        .toMap(TurboLanding::getId, Function.identity());

        ValidationResult<List<TurboLandingWithCountersAndGoals>, Defect> validationResult =
                turboLandingValidationService.
                        validateAddOrUpdateTurbolandings(turboLandings, existingTurboLandingsByIds);

        List<TurboLandingWithCountersAndGoals> validTurbolandings = getValidItems(validationResult);

        if (validTurbolandings.isEmpty()) {
            return MassResult.brokenMassAction(turbolandingIds, validationResult);
        }

        fillModerationParameters(validTurbolandings, existingTurboLandingsByIds);

        addMetrikaGrantsForNewCounters(operatorUid, clientId, validTurbolandings);

        turboLandingRepository.addOrUpdateTurboLandings(shard, validTurbolandings);

        return MassResult.successfulMassAction(turbolandingIds, validationResult);
    }

    /**
     * Обновить  права в Метрике для всех системных счетчиков турболендингов клиента
     * Вызывается при изменении списка представителей клиента, либо при изменении списка пользователей,
     * имеющих доступ к клиенту - изменение менеджера, списка представителей агенства, и.т.п.
     */

    public void refreshTurbolandingMetrikaGrants(long operatorUid, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Set<Long> counters = turboLandingRepository.getSystemTurboLandingMetrikaCounterIdsByClientId(shard,
                clientId);

        updateCounterGrantsService.refreshMetrikaGrants(operatorUid, ImmutableMap.of(clientId, counters));
    }

    public Set<Long> getAttachedBannerIds(ClientId clientId, long turboLandingId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return StreamEx.of(turboLandingRepository
                .getBannerIdsLinkedWithTurboLandingDirectly(shard, clientId, turboLandingId))
                .append(turboLandingRepository
                        .getBannerIdsLinkedWithTurboLandingViaSitelink(shard, clientId, turboLandingId))
                .toSet();
    }

    /**
     * Получает активные (опубликованные, незабаненные) турболендинги клиента из конструктора.
     * @param clientId id клиента
     * @return список турболендингов или {@code null} в случае ошибки
     */
    public @Nullable List<DcTurboLanding> externalGetActiveTurboLandingsOrNull(ClientId clientId) {
        try {
            return turboLandingsClient.getTurboLandings(clientId.asLong(), emptyList());
        } catch (TurboLandingsClientException | TraceInterceptionSemaphoreException e) {
            logger.warn("Got exception while fetching turbolandings", e);
            return null;
        }
    }

    public TurboLanding externalFindTurboLandingByUrl(String url) {
        return externalFindTurboLandingsByUrl(Set.of(url)).get(url);
    }

    /**
     * По заданному списку url запрашивает в конструкторе данные турболендингов, которым эти url принадлежат
     * Возвращает Map 'url из входного списка' -> 'найденный турболендинг'
     * Если по url найти турболендинг не удалась - url в результирующей структуре будет отсутствовать.
     */

    public Map<String, TurboLanding> externalFindTurboLandingsByUrl(Set<String> urls) {
        List<GetIdByUrlResponseItem> tlInfoByUrl = turboLandingsClient.getTurbolandingIdsByUrl(urls);
        if (tlInfoByUrl.isEmpty()) {
            return emptyMap();
        } else {
            Map<String, TurboLanding> result = new HashMap<>();

            shardHelper.groupByShard(tlInfoByUrl, ShardKey.CLIENT_ID, GetIdByUrlResponseItem::getClientId).forEach(
                    (shard, chunk) -> {
                        //Сгруппируем идентификаторы турболендингов по ClientId
                        Map<ClientId, List<Long>> tlIdsByClientId = StreamEx.of(chunk)
                                .mapToEntry(GetIdByUrlResponseItem::getClientId, GetIdByUrlResponseItem::getLandingId)
                                .mapKeys(ClientId::fromLong)
                                .grouping();
                        /*
                        Поклиентно достанем данные турболендингов из базы.
                        Сейчас метод вызывается только из интерфейса, предполагается что число найденных ClientId невелико (в большинстве случаев
                        один).
                        Если метод будет регулярно использоваться для множества клиентов - нужно в репозитории добавить метод,
                        достающий турболендинги нескольких клиентов одним запросом. */
                        Map<Long, TurboLanding> landingsById = EntryStream.of(tlIdsByClientId)
                                .map(idsByClientId -> getClientTurboLandingsbyId(shard, idsByClientId))
                                .nonNull()
                                .flatMap(List::stream)
                                .toMap(TurboLanding::getId, Function.identity());

                        //Добавим найденные лендинги в результирующий Map
                        result.putAll(StreamEx.of(chunk)
                                .mapToEntry(GetIdByUrlResponseItem::getUrl, GetIdByUrlResponseItem::getLandingId)
                                .mapValues(landingsById::get)
                                .nonNullValues()
                                .toMap());
                    }
            );

            return result;
        }
    }

    private List<TurboLanding> getClientTurboLandingsbyId(Integer shard, Map.Entry<ClientId, List<Long>> idsByClientId) {
        ClientId clientId = idsByClientId.getKey();
        List<Long> tlIds = idsByClientId.getValue();

        return turboLandingRepository.getClientTurboLandingsbyId(shard, clientId, tlIds);
    }

    /**
     * Из заданного списка url выбирает те, которые потенциально могут принадлежать турболендингам.
     * Если такие нашлись - запрашивает через Конструктор соответствующие им турболендинги.
     */

    public Map<String, TurboLanding> findTurboLandingsByUrl(Collection<String> urls) {
        List<String> filteredUrls = StreamEx.of(urls).nonNull()
                .filter(this::isTurboLandingUrl)
                .toList();
        return filteredUrls.isEmpty() ? emptyMap() : externalFindTurboLandingsByUrl(new HashSet<>(filteredUrls));
    }

    public boolean isTurboLandingUrl(String url) {
        return TURBO_LANDING_URL_PATTERN.matcher(url.toLowerCase()).find();
    }

    private void addMetrikaGrantsForNewCounters(long operatorUid, ClientId clientId,
                                                List<TurboLandingWithCountersAndGoals> turboLandings) {
        if (turboLandings.isEmpty()) {
            return;
        }
        Set<Long> counters = getSystemCountersFromTurboLandings(turboLandings);

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Set<Long> existingCounters = turboLandingRepository
                .getSystemTurboLandingMetrikaCounterIdsByClientId(shard, clientId);

        Set<Long> newCounters = StreamEx.of(counters)
                .filter(c -> !existingCounters.contains(c))
                .toSet();
        if (!newCounters.isEmpty()) {
            updateCounterGrantsService.refreshMetrikaGrants(operatorUid, ImmutableMap.of(clientId, newCounters));
        }
    }

    private Set<Long> getSystemCountersFromTurboLandings(List<TurboLandingWithCountersAndGoals> turboLandings) {
        return StreamEx.of(turboLandings)
                .nonNull().map(TurboLandingWithCountersAndGoals::getCounters)
                .nonNull().flatMap(counters -> counters.stream())
                .filter(c -> !Boolean.TRUE.equals(c.getIsUserCounter()))
                .map(TurboLandingMetrikaCounter::getId)
                .nonNull()
                .toSet();
    }

    public List<TurboLanding> getTurbolandigsByClientId(ClientId clientId) {
        return turboLandingRepository.getClientTurbolandings(shardHelper.getShardByClientId(clientId),
                clientId, null, LimitOffset.maxLimited());
    }
}
