package ru.yandex.direct.core.entity.turbolanding.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.turbolanding.container.UpdateCounterGrantsParams;
import ru.yandex.direct.core.entity.turbolanding.container.UpdateCounterGrantsParamsItem;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.metrika.client.MetrikaApiError;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.model.response.UpdateCounterGrantsResponse;
import ru.yandex.direct.rbac.ClientPerminfo;
import ru.yandex.direct.rbac.PpcRbac;
import ru.yandex.direct.rbac.model.Representative;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.UPDATE_COUNTER_GRANTS_JOB;

@Service
public class UpdateCounterGrantsService {
    public static final Set<String> ERROR_TYPES_WITHOUT_RETRY = Set.of("invalid_parameter", "not_found");
    private static final int MAX_CHUNK_SIZE = 20;

    private final ShardHelper shardHelper;
    private final MetrikaClient metrikaClient;
    private final DbQueueRepository dbQueueRepository;
    private final PpcRbac ppcRbac;
    private final ClientService clientService;


    @Autowired
    public UpdateCounterGrantsService(
            ShardHelper shardHelper,
            MetrikaClient metrikaClient,
            DbQueueRepository dbQueueRepository,
            PpcRbac ppcRbac,
            ClientService clientService) {
        this.shardHelper = shardHelper;

        this.metrikaClient = metrikaClient;
        this.dbQueueRepository = dbQueueRepository;
        this.ppcRbac = ppcRbac;
        this.clientService = clientService;
    }

    /**
     * Установить для переданных счетчиков указанные списки доступа
     * (должно вызываться только из UpdateMetrikaCounterGrantsJob)
     *
     * @return информация для переотправки счетчиков из джобы:
     * id счетчиков, не нуждающихся в повторной отправке (либо успешно обработаны, либо повторная отправка
     * не даст результатов)
     */
    public UpdateCounterGrantsResultForRetry updateCounterGrants(Map<Long, Set<String>> userLoginsByCounterId) {
        if (userLoginsByCounterId.isEmpty()) {
            return new UpdateCounterGrantsResultForRetry(emptySet(), false);
        }

        Map<Long, Boolean> shouldNotRetryByCounterId = new HashMap<>();

        for (Long counterId : userLoginsByCounterId.keySet()) {
            UpdateCounterGrantsResponse response = metrikaClient.updateCounterGrants(counterId,
                    userLoginsByCounterId.get(counterId));

            //ставим на переотправку в случае наличия только ошибок, которые могут исправиться при перезапуске
            boolean shouldNotRetry = response.isSuccessful() ||
                    (response.getMetrikaErrorResponse() != null &&
                            response.getMetrikaErrorResponse().getErrors().stream()
                                    .anyMatch(UpdateCounterGrantsService::shouldNotRetry));
            shouldNotRetryByCounterId.put(counterId, shouldNotRetry);
        }

        Set<Long> counterIdsThatShouldNotRetry = EntryStream.of(shouldNotRetryByCounterId)
                .filterValues(shouldNotRetry -> shouldNotRetry)
                .keys()
                .toSet();

        return new UpdateCounterGrantsResultForRetry(counterIdsThatShouldNotRetry,
                counterIdsThatShouldNotRetry.size() != shouldNotRetryByCounterId.size());
    }

    private static boolean shouldNotRetry(MetrikaApiError error) {
        return ERROR_TYPES_WITHOUT_RETRY.contains(error.getErrorType());
    }

    public static class UpdateCounterGrantsResultForRetry {
        private final Set<Long> counterIdsThatShouldNotRetry;
        private final boolean needToRetry;

        public UpdateCounterGrantsResultForRetry(Set<Long> counterIdsThatShouldNotRetry, boolean needToRetry) {
            this.counterIdsThatShouldNotRetry = counterIdsThatShouldNotRetry;
            this.needToRetry = needToRetry;
        }

        public Set<Long> getCounterIdsThatShouldNotRetry() {
            return counterIdsThatShouldNotRetry;
        }

        public boolean isNeedToRetry() {
            return needToRetry;
        }
    }

    /**
     * Добавить в очередь задания на установку для указанных счетчиков метрики прав доступа,
     * возвращает id добавленных заданий (нужны для тестов)
     */
    public List<Long> addCountersToResyncQueue(Long operatorUid, Map<Long, List<Long>> userIdsByCounterId) {
        if (userIdsByCounterId.isEmpty()) {
            return emptyList();
        }

        ClientId operatorClientId = ClientId.fromLong(shardHelper.getClientIdByUid(operatorUid));
        int shard = shardHelper.getShardByClientIdStrictly(operatorClientId);

        List<Long> counterIds = new ArrayList<>(userIdsByCounterId.keySet());
        List<Long> jobIds = new ArrayList<>();

        for (List<Long> chunk : Lists.partition(counterIds, MAX_CHUNK_SIZE)) {

            List<UpdateCounterGrantsParamsItem> items = StreamEx.of(chunk)
                    .map(counterId -> new UpdateCounterGrantsParamsItem()
                            .withCounterId(counterId)
                            .withUserIds(userIdsByCounterId.get(counterId)))
                    .toList();

            UpdateCounterGrantsParams params = new UpdateCounterGrantsParams()
                    .withItems(items);

            Long jobId = dbQueueRepository
                    .insertJob(shard, UPDATE_COUNTER_GRANTS_JOB, operatorClientId, operatorUid, params)
                    .getId();

            jobIds.add(jobId);
        }
        return jobIds;
    }

    public List<Long> refreshMetrikaGrants(Long operatorUid, Map<ClientId, Set<Long>> counterIdsByClientId) {
        Set<ClientId> clientIds = ImmutableSet.copyOf(counterIdsByClientId.keySet());
        Map<ClientId, Set<Long>> relatedUserIdsByClientId = getRelatedUsersByClientId(clientIds);

        Map<Long, List<Long>> userIdsByCounterId = EntryStream.of(relatedUserIdsByClientId)
                .filterKeys(counterIdsByClientId::containsKey)
                .flatMapKeys(clientId -> counterIdsByClientId.get(clientId).stream())
                .flatMapValues(Collection::stream)
                .grouping();

        return addCountersToResyncQueue(operatorUid, userIdsByCounterId);
    }

    /**
     * По каждому из переданных client_id получить список uid пользователей, имеющих доступ к ресурсам клиента
     * (представители клиента, менеджеры, представители агенства)
     */
    public Map<ClientId, Set<Long>> getRelatedUsersByClientId(Collection<ClientId> clientIds) {
        if (clientIds.isEmpty()) {
            return emptyMap();
        }

        Collection<Representative> clientRepresentatives = ppcRbac.massGetClientRepresentatives(clientIds);

        Map<ClientId, List<Long>> clientUidsByClientId = StreamEx.of(clientRepresentatives)
                .mapToEntry(Representative::getClientId, Representative::getUserId)
                .grouping();

        Map<ClientId, Optional<ClientPerminfo>> clients = ppcRbac.getClientsPerminfo(clientIds);

        Set<ClientId> agencyClientIds = StreamEx.ofValues(clients)
                .map(client -> client.map(ClientPerminfo::agencyClientId).orElse(null))
                .nonNull()
                .toSet();
        Collection<Representative> agencyRepresentatives = ppcRbac.massGetClientRepresentatives(agencyClientIds);

        Map<ClientId, List<Long>> mainAgencyUidsByAgencyId = StreamEx.of(agencyRepresentatives)
                .filter(rep -> !rep.isLimited())
                .mapToEntry(Representative::getClientId, Representative::getUserId)
                .grouping();

        Set<Long> limitedAgencyUids = StreamEx.of(agencyRepresentatives)
                .filter(Representative::isLimited)
                .map(Representative::getUserId)
                .toSet();

        Map<Long, Set<Long>> clientsManagersFromCampaigns = clientService.getClientsManagersFromCampaigns(clientIds);

        return EntryStream.of(clients)
                .mapValues(client -> {

                    List<Long> clientUids = client.map(ClientPerminfo::clientId)
                            .map(clientUidsByClientId::get)
                            .orElse(emptyList());

                    Set<Long> managerUids = client
                            .map(ClientPerminfo::clientId)
                            .map(ClientId::asLong)
                            .map(clientsManagersFromCampaigns::get)
                            .orElse(emptySet());

                    List<Long> mainAgencyUids = client.map(ClientPerminfo::agencyClientId)
                            .map(mainAgencyUidsByAgencyId::get)
                            .orElse(emptyList());

                    Set<Long> filteredLimitedAgencyUids = client.map(ClientPerminfo::agencyUids)
                            .orElse(emptySet())
                            .stream().filter(limitedAgencyUids::contains)
                            .collect(Collectors.toSet());

                    return StreamEx.of(clientUids)
                            .append(managerUids)
                            .append(mainAgencyUids)
                            .append(filteredLimitedAgencyUids)
                            .nonNull()
                            .toSet();
                })
                .removeValues(Set::isEmpty)
                .toMap();
    }
}
