package ru.yandex.direct.core.entity.turbolanding.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.turbolanding.model.StatusModerateForCpa;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingWithCountersAndGoals;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.turbolanding.service.validation.defects.TurbolandingDefects.inconsistentStateOfModerationStatus;
import static ru.yandex.direct.core.entity.turbolanding.service.validation.defects.TurbolandingDefects.moderationRequiredAttributeCantBeChanged;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;

@Service
public class TurboLandingValidationService {

    private static final Set<StatusModerateForCpa> ALLOWED_INCOMING_STATUS_MODERATE_FOR_CPA =
            Set.of(StatusModerateForCpa.YES, StatusModerateForCpa.NO);

    public TurboLandingValidationService() {
    }

    private static ValidationResult<TurboLandingWithCountersAndGoals, Defect> validateTurbolanding(
            TurboLandingWithCountersAndGoals turbolanding,
            Map<Long, TurboLanding> existedTurbolandingsByIds) {

        ModelItemValidationBuilder<TurboLandingWithCountersAndGoals> vb = ModelItemValidationBuilder.of(turbolanding);
        if (!existedTurbolandingsByIds.containsKey(turbolanding.getId())) {
            vb.check(Constraint.fromPredicate(tl -> tl.getStatusModerateForCpa() == StatusModerateForCpa.READY,
                    inconsistentStateOfModerationStatus()),
                    When.isTrue(turbolanding.getIsCpaModerationRequired()));
            return vb.getResult();
        }

        var savedTurbolanding = existedTurbolandingsByIds.get(turbolanding.getId());

        vb.item(TurboLanding.IS_CPA_MODERATION_REQUIRED)
                .check(Constraint.fromPredicate(
                        isCpaModerationRequired -> isCpaModerationRequired == savedTurbolanding.getIsCpaModerationRequired(),
                        moderationRequiredAttributeCantBeChanged()));
        vb.item(TurboLanding.VERSION)
                .check(notLessThan(savedTurbolanding.getVersion()));
        vb.check(Constraint.fromPredicate(tl -> moderationStatusSatisfy(tl, savedTurbolanding),
                inconsistentStateOfModerationStatus()));

        return vb.getResult();
    }

    private static boolean moderationStatusSatisfy(TurboLanding newTurbo,
                                                   TurboLanding savedTurbo) {
        var newModerationStatus = newTurbo.getStatusModerateForCpa();

        return ((newTurbo.getVersion().compareTo(savedTurbo.getVersion()) > 0
                && (newModerationStatus == StatusModerateForCpa.READY))
                //обновление текущей турбостраницы без изменения статуса модерации (DIRECT-117772)
                || (newTurbo.getVersion().equals(savedTurbo.getVersion()) && newModerationStatus == null)
                //обновление текущией турбостраницы cо статусом модерации
                || (newTurbo.getVersion().equals(savedTurbo.getVersion()) && newModerationStatus != null
                && ALLOWED_INCOMING_STATUS_MODERATE_FOR_CPA.contains(newModerationStatus)
                && isSavedModerationStatusSatisfy(newModerationStatus, savedTurbo.getStatusModerateForCpa()))
        );
    }

    private static boolean isSavedModerationStatusSatisfy(StatusModerateForCpa newModerationStatus,
                                                          StatusModerateForCpa savedModerationStatus) {
        return savedModerationStatus == newModerationStatus || savedModerationStatus == StatusModerateForCpa.SENT;
    }

    public ValidationResult<List<TurboLandingWithCountersAndGoals>, Defect> validateAddOrUpdateTurbolandings(List<TurboLandingWithCountersAndGoals> turboLandings,
                                                                                                             Map<Long, TurboLanding> existedTurbolandingsByIds) {
        ListValidationBuilder<TurboLandingWithCountersAndGoals, Defect> lvb = ListValidationBuilder.of(turboLandings);

        lvb.checkEachBy(tl -> validateTurbolanding(tl, existedTurbolandingsByIds));
        return lvb.getResult();
    }

}
