package ru.yandex.direct.core.entity.uac

import com.fasterxml.jackson.annotation.JsonInclude
import com.fasterxml.jackson.core.type.TypeReference
import java.math.BigDecimal
import java.math.RoundingMode
import java.util.EnumSet
import org.apache.commons.collections4.CollectionUtils
import org.slf4j.LoggerFactory
import org.springframework.http.HttpStatus
import org.springframework.http.ResponseEntity
import ru.yandex.direct.core.entity.bidmodifier.AgeType
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnly
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnlyAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTV
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTVAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTabletAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier
import ru.yandex.direct.core.entity.bidmodifier.GenderType
import ru.yandex.direct.core.entity.bidmodifier.OsType
import ru.yandex.direct.core.entity.uac.converter.UacBidModifiersConverter.toCoreInventoryType
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.DeviceType
import ru.yandex.direct.core.entity.uac.model.InventoryType
import ru.yandex.direct.core.entity.uac.model.Socdem
import ru.yandex.direct.core.entity.uac.model.UacComplexBidModifier
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.utils.mapToSet
import ru.yandex.direct.utils.model.UrlParts
import ru.yandex.direct.validation.constraint.StringConstraints.isValidHref

object UacCommonUtils {
    private val logger = LoggerFactory.getLogger(UacCommonUtils::class.java)

    const val CREATIVE_ID_KEY = "creative_id"
    const val CREATIVE_TYPE_KEY = "creative_type"

    fun uacErrorResponse(
        title: String? = null,
        description: String? = null,
        code: Int? = null,
        status: HttpStatus
    ): ResponseEntity<Any> {
        if (title == null && description == null && code == null) {
            logger.error("invalid request for uac campaigns with status $status")
            return ResponseEntity(status)
        }
        val response = JsonUtils.toJson(UacErrorResponse(title, description, code))
        logger.error("invalid request for uac campaigns with response $response and status $status")
        return ResponseEntity(response, status)
    }

    // https://a.yandex-team.ru/arc_vcs/direct/libs-internal/grid-processing/src/main/java/ru/yandex/direct/grid/processing/service/campaign/converter/UcCampaignConverter.java?rev=eedd1e6408#L555
    fun getComplexBidModifier(
        uacComplexBidModifier: UacComplexBidModifier,
        campaignId: Long
    ): ComplexBidModifier {
        val complexBidModifier = ComplexBidModifier()
        if (uacComplexBidModifier.advType == AdvType.CPM_BANNER) {
            addDeviceTypeBidModifiersForCpmBannerCampaign(complexBidModifier, campaignId,
                uacComplexBidModifier.deviceTypes, uacComplexBidModifier.isSmartTVEnabled)
        } else {
            // Соцдем как корректировки не используется в медийных кампаниях
            complexBidModifier.withDemographyModifier(getSocdemBidModifier(uacComplexBidModifier.socdem))
            addDeviceTypeBidModifiers(
                complexBidModifier,
                campaignId,
                uacComplexBidModifier.deviceTypes,
                uacComplexBidModifier.isTabletModifierEnabled)
        }
        return complexBidModifier
    }

    private fun addDeviceTypeBidModifiers(complexBidModifier: ComplexBidModifier,
                                          campaignId: Long,
                                          deviceTypes: Set<DeviceType>?,
                                          tabletBidModifierEnabled: Boolean) {

        if (CollectionUtils.isEmpty(deviceTypes) || deviceTypes!!.contains(DeviceType.ALL)) {
            return
        }

        if (tabletBidModifierEnabled) {
            if (!deviceTypes.contains(DeviceType.DESKTOP)) {
                val bidModifierDesktopOnly = BidModifierDesktopOnly()
                    .withCampaignId(campaignId)
                    .withEnabled(true)
                    .withDesktopOnlyAdjustment(BidModifierDesktopOnlyAdjustment().withPercent(0))
                    .withType(BidModifierType.DESKTOP_ONLY_MULTIPLIER)
                complexBidModifier.withDesktopOnlyModifier(bidModifierDesktopOnly)
            }
            if (!deviceTypes.contains(DeviceType.TABLET)) {
                val bidModifierTablet = BidModifierTablet()
                    .withCampaignId(campaignId)
                    .withEnabled(true)
                    .withTabletAdjustment(BidModifierTabletAdjustment().withPercent(0))
                    .withType(BidModifierType.TABLET_MULTIPLIER)
                complexBidModifier.withTabletModifier(bidModifierTablet)
            }
        } else {
            if (!deviceTypes.contains(DeviceType.DESKTOP) && !deviceTypes.contains(DeviceType.TABLET)) {
                val bidModifierDesktop = BidModifierDesktop()
                    .withCampaignId(campaignId)
                    .withEnabled(true)
                    .withDesktopAdjustment(BidModifierDesktopAdjustment().withPercent(0))
                    .withType(BidModifierType.DESKTOP_MULTIPLIER)
                complexBidModifier.withDesktopModifier(bidModifierDesktop)
            }
        }

        if (!deviceTypes.contains(DeviceType.PHONE)) {
            val bidModifierMobile = BidModifierMobile()
                .withCampaignId(campaignId)
                .withEnabled(true)
                .withMobileAdjustment(BidModifierMobileAdjustment().withPercent(0))
                .withType(BidModifierType.MOBILE_MULTIPLIER)
            complexBidModifier.withMobileModifier(bidModifierMobile)
        }
    }

    private fun addDeviceTypeBidModifiersForCpmBannerCampaign(
        complexBidModifier: ComplexBidModifier,
        campaignId: Long,
        deviceTypes: Set<DeviceType>?,
        isSmartTVEnabled: Boolean = false,
    ) {
        if (CollectionUtils.isEmpty(deviceTypes) || deviceTypes!!.contains(DeviceType.ALL)) {
            return
        }
        if (!deviceTypes.contains(DeviceType.DESKTOP) && !deviceTypes.contains(DeviceType.TABLET)) {
            val bidModifierDesktop = BidModifierDesktop()
                .withCampaignId(campaignId)
                .withEnabled(true)
                .withDesktopAdjustment(BidModifierDesktopAdjustment().withPercent(0))
                .withType(BidModifierType.DESKTOP_MULTIPLIER)
            complexBidModifier.withDesktopModifier(bidModifierDesktop)
        }
        if (isSmartTVEnabled && !deviceTypes.contains(DeviceType.SMART_TV)) {
            val bidModifierSmartTV = BidModifierSmartTV()
                .withCampaignId(campaignId)
                .withEnabled(true)
                .withSmartTVAdjustment(BidModifierSmartTVAdjustment().withPercent(0))
                .withType(BidModifierType.SMARTTV_MULTIPLIER)
            complexBidModifier.withSmartTVModifier(bidModifierSmartTV)
        }

        val bidModifierMobile = BidModifierMobile()
            .withCampaignId(campaignId)
            .withEnabled(true)
            .withMobileAdjustment(BidModifierMobileAdjustment().withPercent(0))
            .withType(BidModifierType.MOBILE_MULTIPLIER)
        if (!deviceTypes.contains(DeviceType.PHONE)
            && !deviceTypes.contains(DeviceType.PHONE_IOS)
            && !deviceTypes.contains(DeviceType.PHONE_ANDROID)) {

            complexBidModifier.withMobileModifier(bidModifierMobile)

        } else if (deviceTypes.contains(DeviceType.PHONE_ANDROID)) {

            bidModifierMobile.mobileAdjustment.osType = OsType.IOS
            complexBidModifier.withMobileModifier(bidModifierMobile)

        } else if (deviceTypes.contains(DeviceType.PHONE_IOS)) {

            bidModifierMobile.mobileAdjustment.osType = OsType.ANDROID
            complexBidModifier.withMobileModifier(bidModifierMobile)

        }
    }

    private fun getSocdemBidModifier(socdem: Socdem?): BidModifierDemographics? {
        if (socdem == null) {
            return null
        }
        val bmd = BidModifierDemographics()
            .withType(BidModifierType.DEMOGRAPHY_MULTIPLIER)
            .withEnabled(true)
        var genders = EnumSet.noneOf(GenderType::class.java)
        genders!!.addAll(socdem.genders.map { GenderType.valueOf(it.name) })
        var invGenders = EnumSet.complementOf(genders)
        var emptyGenders = false
        if (genders.isEmpty() || invGenders!!.isEmpty()) {
            emptyGenders = true
            genders = null
            invGenders = null
        }
        val lowerAgeTypeOrdinal = socdem.ageLower.getLowerIntervalAboveAgePoint().ordinal
        val upperAgeTypeOrdinal = socdem.ageUpper.getUpperIntervalUnderAgePoint().ordinal

        val ages = EnumSet.noneOf(AgeType::class.java)
        ages.addAll(
            AgeType.values().filter { it.ordinal >= lowerAgeTypeOrdinal && it.ordinal <= upperAgeTypeOrdinal }
        )
        var invAges = EnumSet.complementOf(ages)
        // период 45+ больше не используется, вместо него 45-55 + 55+
        invAges!!.remove(AgeType._45_)
        // неизвестный возраст не поддерживается в веб интерфейсе, поэтому не исключаем его корректировкой
        invAges.remove(AgeType.UNKNOWN)
        var emptyAges = false
        if (ages.isEmpty() || invAges.isEmpty()) {
            emptyAges = true
            invAges = null
        }
        if (emptyGenders && emptyAges) {
            return null
        }
        val adjList = ArrayList<BidModifierDemographicsAdjustment>()
        if (emptyGenders) {
            // таргетируемся на все гендеры определённых возрастов
            for (age in invAges) {
                adjList.add(BidModifierDemographicsAdjustment()
                    .withGender(null)
                    .withAge(age)
                    .withPercent(0))
            }
        } else {
            for (gender in invGenders!!) {
                // на этот гендер не таргетируемся в любом возрасте
                adjList.add(BidModifierDemographicsAdjustment()
                    .withGender(gender)
                    .withAge(null)
                    .withPercent(0))
            }
            if (!emptyAges) {
                for (gender in genders) {
                    // есть таргетинг одновременно на гендер и на диапазон возрастов
                    // выключаем показы у возрастов не включенных в таргет (гендеры уже выключили выше)
                    for (age in invAges) {
                        adjList.add(BidModifierDemographicsAdjustment()
                            .withGender(gender)
                            .withAge(age)
                            .withPercent(0))
                    }
                }
            }
        }
        bmd.demographicsAdjustments = adjList
        return bmd
    }

    /**
     * Приклеивает к ссылке трекинговые параметры. Если ссылка невалидная, возвращает ее же, игнорируя параметры.
     * Существующие трек. параметры игнорируются
     */
    @JvmStatic
    fun getHrefWithTrackingParams(href: String, trackingParams: String?): String {
        return if (trackingParams == null || !isValidHref(href) || trackingParams.isBlank()) href else {
            val hrefUrlParts = UrlParts.fromUrl(href)
            val parsedTrackingParams = UrlParts.parseParameters(trackingParams)
            val trackingParamsKeySet = parsedTrackingParams.mapToSet { it.key }

            val filteredParamsOnHref = hrefUrlParts.parameters?.filter { !trackingParamsKeySet.contains(it.key) }

            hrefUrlParts.toBuilder().withParameters(parsedTrackingParams).addParameters(filteredParamsOnHref).build().toUrl()
        }
    }

    fun getInventoryTypesBidModifier(inventoryTypes: Set<InventoryType>?): BidModifierInventory? {
        if (CollectionUtils.isEmpty(inventoryTypes) || inventoryTypes!!.contains(InventoryType.ALL)) {
            return null
        }

        val inventoryTypesForZero: List<InventoryType> = InventoryType.values().asList()
            .filter { inventoryType: InventoryType ->
                inventoryType != InventoryType.ALL && !inventoryTypes.contains(inventoryType)
            }

        if (inventoryTypesForZero.isEmpty()) {
            return null
        }

        val adjustments = inventoryTypesForZero.map {
            BidModifierInventoryAdjustment()
                .withInventoryType(toCoreInventoryType(it))
                .withPercent(0)
        }

        return BidModifierInventory()
            .withType(BidModifierType.INVENTORY_MULTIPLIER)
            .withEnabled(true)
            .withInventoryAdjustments(adjustments)
    }

    fun mapFromJson(json: String): Map<String, Any> {
        return JsonUtils.fromJson(json,
            object : TypeReference<Map<String, Any>>() {}
        )
    }
}

@JsonInclude(JsonInclude.Include.NON_NULL)
data class UacErrorResponse(
    val title: String?,
    val description: String?,
    val code: Int?)

fun List<BigDecimal>.average(count: Int): BigDecimal {
    val totalCost = this
        .sumOf { it }

    return if (count == 0) BigDecimal.ZERO else totalCost.divide(BigDecimal(count), RoundingMode.HALF_UP)
}
