package ru.yandex.direct.core.entity.uac.converter

import ru.yandex.direct.core.entity.bidmodifier.AgeType
import ru.yandex.direct.core.entity.bidmodifier.BidModifier
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierGeo
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRegionalAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargeting
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilter
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilterAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType
import ru.yandex.direct.core.entity.bidmodifier.GenderType
import ru.yandex.direct.core.entity.uac.converter.UacRegionNamesConverter.regionIdsToNames
import ru.yandex.direct.core.entity.uac.model.AgePoint
import ru.yandex.direct.core.entity.uac.model.Gender
import ru.yandex.direct.core.entity.uac.model.InventoryType
import ru.yandex.direct.core.entity.uac.model.UacAdjustment
import ru.yandex.direct.core.entity.uac.model.UacAdjustmentRequest
import ru.yandex.direct.regions.GeoTree

object UacBidModifiersConverter {
    fun toBidModifiers(
        uacAdjustments: List<UacAdjustmentRequest>?,
        retargetingConditionId: Long?,
        campaignId: Long,
        isSearchRetargetingEnabled: Boolean,
    ): List<BidModifier> {
        val bidModifiers = mutableListOf<BidModifier>()
        val geoModifier = getRegionalModifier(uacAdjustments, campaignId)
        if (geoModifier != null) {
            bidModifiers.add(geoModifier)
        }

        val demographicsModifier = getDemographicsModifier(uacAdjustments, campaignId)
        if (demographicsModifier != null) {
            bidModifiers.add(demographicsModifier)
        }


        val retargetingModifier = getRetargetingModifier(uacAdjustments, campaignId)
        if (retargetingModifier != null) {
            bidModifiers.add(retargetingModifier)
        }
        if (retargetingConditionId != null && isSearchRetargetingEnabled) {
            val bidModifierRetargetingFilterAdjustment = BidModifierRetargetingFilterAdjustment()
                .withRetargetingConditionId(retargetingConditionId)
                .withPercent(0)
            bidModifiers.add(
                BidModifierRetargetingFilter()
                    .withCampaignId(campaignId)
                    .withRetargetingAdjustments(listOf(bidModifierRetargetingFilterAdjustment))
                    .withType(BidModifierType.RETARGETING_FILTER)
            )
        }
        return bidModifiers
    }

    private fun getRegionalModifier(
        uacAdjustments: List<UacAdjustmentRequest>?,
        campaignId: Long
    ): BidModifierGeo? {
        val geoAdjustments = uacAdjustments
            ?.filter { it.region != null }
            ?.map { BidModifierRegionalAdjustment()
                .withRegionId(it.region)
                //Повышение на 100 требуется, т.к. с фронта приходит значение "на сколько процентов повысить/понизить"
                .withPercent(it.percent + 100)
                .withHidden(false) }
        return if (!geoAdjustments.isNullOrEmpty()) {
            BidModifierGeo()
                .withCampaignId(campaignId)
                .withType(BidModifierType.GEO_MULTIPLIER)
                .withEnabled(true)
                .withRegionalAdjustments(geoAdjustments)
        } else {
            null
        }
    }

    private fun getDemographicsModifier(
        uacAdjustments: List<UacAdjustmentRequest>?,
        campaignId: Long
    ): BidModifierDemographics? {
        val demographicsAdjustments = uacAdjustments
            ?.filter { it.age != null || it.gender != null }
            ?.map { BidModifierDemographicsAdjustment()
                .withAge(it.age?.getLowerIntervalAboveAgePoint())
                .withGender(it.gender?.getGenderType())
                .withPercent(it.percent + 100) }
        return if (!demographicsAdjustments.isNullOrEmpty()) {
            BidModifierDemographics()
                .withCampaignId(campaignId)
                .withType(BidModifierType.DEMOGRAPHY_MULTIPLIER)
                .withEnabled(true)
                .withDemographicsAdjustments(demographicsAdjustments)
        } else {
            null
        }
    }

    private fun getRetargetingModifier(
        uacAdjustments: List<UacAdjustmentRequest>?,
        campaignId: Long
    ): BidModifierRetargeting? {
        val retargetingAdjustments = uacAdjustments
            ?.filter { it.retargetingConditionId != null }
            ?.map { BidModifierRetargetingAdjustment()
                .withRetargetingConditionId(it.retargetingConditionId)
                .withPercent(it.percent + 100) }
       return if (!retargetingAdjustments.isNullOrEmpty()) {
            BidModifierRetargeting()
                .withCampaignId(campaignId)
                .withType(BidModifierType.RETARGETING_MULTIPLIER)
                .withEnabled(true)
                .withRetargetingAdjustments(retargetingAdjustments)
        } else {
            null
       }
    }

    fun toUacAdjustments(bidModifier: List<BidModifier>?, geoTree: GeoTree): List<UacAdjustment>? {
        return bidModifier?.flatMap { it ->
            when (it) {
                is BidModifierGeo -> toUacGeoAdjustments(it, geoTree)
                is BidModifierDemographics -> toUacDemographicsAdjustments(it)
                is BidModifierRetargeting -> toUacRetargetingAdjustments(it)
                else -> throw IllegalStateException("Incorrect bid modifier")
            }
        }
    }

    private fun toUacGeoAdjustments(modifier: BidModifierGeo, geoTree: GeoTree): List<UacAdjustment> {
        return modifier.regionalAdjustments
            .filter { it.hidden == false }
            .map {
                UacAdjustment(
                    region = it.regionId,
                    regionName = regionIdsToNames(geoTree, listOf(it.regionId))[0],
                    gender = null,
                    age = null,
                    percent = it.percent - 100,
                    retargetingConditionId = null,
                )
            }
    }

    private fun toUacDemographicsAdjustments(modifier: BidModifierDemographics): List<UacAdjustment> {
        return modifier.demographicsAdjustments.map {
            UacAdjustment(
                region = null,
                regionName = null,
                gender = toGender(it.gender),
                age = toAgePoint(it.age),
                percent = it.percent - 100,
                retargetingConditionId = null
            )
        }
    }

    private fun toUacRetargetingAdjustments(modifier: BidModifierRetargeting): List<UacAdjustment> {
        return modifier.retargetingAdjustments.map {
            UacAdjustment(
                region = null,
                regionName = null,
                gender = null,
                age = null,
                percent = it.percent - 100,
                retargetingConditionId = it.retargetingConditionId,
            )
        }
    }

    fun toUacAdjustments(uacAdjustmentRequests: List<UacAdjustmentRequest>?): List<UacAdjustment>? {
        return uacAdjustmentRequests?.map { UacAdjustment(
                region = it.region,
                regionName = null,
                gender = it.gender,
                age = it.age,
                percent = it.percent,
                retargetingConditionId = it.retargetingConditionId,
            ) }
    }

    private fun toAgePoint(ageType: AgeType?): AgePoint? {
        return when (ageType) {
            AgeType._0_17 -> AgePoint.AGE_0
            AgeType._18_24 -> AgePoint.AGE_18
            AgeType._25_34 -> AgePoint.AGE_25
            AgeType._35_44 -> AgePoint.AGE_35
            AgeType._45_54 -> AgePoint.AGE_45
            AgeType._55_ -> AgePoint.AGE_55
            else -> null
        }
    }

    private fun toGender(genderType: GenderType?): Gender? {
        return when (genderType) {
            GenderType.MALE -> Gender.MALE
            GenderType.FEMALE -> Gender.FEMALE
            else -> null
        }
    }

    fun toCoreInventoryType(uacInventoryType: InventoryType): ru.yandex.direct.core.entity.bidmodifier.InventoryType {
        return when (uacInventoryType) {
            InventoryType.INAPP -> ru.yandex.direct.core.entity.bidmodifier.InventoryType.INAPP
            InventoryType.INPAGE -> ru.yandex.direct.core.entity.bidmodifier.InventoryType.INPAGE
            InventoryType.INSTREAM -> ru.yandex.direct.core.entity.bidmodifier.InventoryType.INSTREAM_WEB
            InventoryType.REWARDED -> ru.yandex.direct.core.entity.bidmodifier.InventoryType.REWARDED
            else -> throw IllegalStateException("Illegal inventoryType $uacInventoryType")
        }
    }

    fun toUacInventoryType(coreInventoryType: ru.yandex.direct.core.entity.bidmodifier.InventoryType): InventoryType {
        return when (coreInventoryType) {
            ru.yandex.direct.core.entity.bidmodifier.InventoryType.INAPP -> InventoryType.INAPP
            ru.yandex.direct.core.entity.bidmodifier.InventoryType.INPAGE -> InventoryType.INPAGE
            ru.yandex.direct.core.entity.bidmodifier.InventoryType.INSTREAM_WEB -> InventoryType.INSTREAM
            ru.yandex.direct.core.entity.bidmodifier.InventoryType.REWARDED -> InventoryType.REWARDED
            else -> throw IllegalStateException("Illegal inventoryType $coreInventoryType")
        }
    }
}
