package ru.yandex.direct.core.entity.uac.converter

import com.fasterxml.jackson.core.type.TypeReference
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetTab
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule
import ru.yandex.direct.core.entity.dynamictextadtarget.utils.DynamicTextAdTargetHashUtils
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilter
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilterCondition
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilterTab
import ru.yandex.direct.core.entity.performancefilter.model.Operator
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterTab
import ru.yandex.direct.core.entity.performancefilter.model.TargetFunnel
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatParser
import ru.yandex.direct.core.entity.uac.model.UacFeedFilter
import ru.yandex.direct.core.entity.uac.model.UacFeedFilterCondition
import ru.yandex.direct.core.entity.uac.model.UacFeedFilterOperator
import ru.yandex.direct.utils.JsonUtils

object UacEcomConverter {
    fun List<UacFeedFilter>.toDynamicFilters(
            groupId: Long,
            name: String,
            feed: Feed,
            filterSchema: FilterSchema): List<DynamicFeedAdTarget> {
        return this.map { it.toDynamicFilter(groupId, name, feed, filterSchema) }
    }

    private fun UacFeedFilter.toDynamicFilter(
            groupId: Long,
            name: String,
            feed: Feed,
            filterSchema: FilterSchema): DynamicFeedAdTarget {
        return DynamicFeedAdTarget()
                .withAdGroupId(groupId)
                .withConditionName(name)
                .withTab(DynamicAdTargetTab.ALL_PRODUCTS)
                .withIsSuspended(false)
                .withBusinessType(feed.businessType)
                .withFeedType(feed.feedType)
                .withCondition(this.conditions.toDynamicConditions(filterSchema))
                .withConditionHash(DynamicTextAdTargetHashUtils.getHashForDynamicFeedRules(listOf()))
    }

    private fun List<UacFeedFilterCondition>.toDynamicConditions(filterSchema: FilterSchema): List<DynamicFeedRule<Any>> {
        val rules = this.map { it.toDynamicCondition() }
        PerformanceFilterConditionDBFormatParser.setParsedValue(filterSchema, rules)
        return rules
    }

    private fun UacFeedFilterCondition.toDynamicCondition() =
            DynamicFeedRule<Any>(this.field, this.operator.toOperator(), this.values?.toFeedFilterValue() ?: this.value)

    fun DynamicFeedAdTarget.toUacFeedFiltersConditions() = this.condition.map {
            UacFeedFilterCondition(it.fieldName, it.operator.toUacFeedFilterOperator(), it.stringValue, it.stringValue.toFeedFilterValues())
        }.toSet()

    private fun UacFeedFilterOperator.toOperator(): Operator = when(this) {
        UacFeedFilterOperator.GREATER -> Operator.GREATER
        UacFeedFilterOperator.LESS -> Operator.LESS
        UacFeedFilterOperator.EQUALS -> Operator.EQUALS
        UacFeedFilterOperator.RANGE -> Operator.RANGE
        UacFeedFilterOperator.CONTAINS -> Operator.CONTAINS
        UacFeedFilterOperator.NOT_CONTAINS -> Operator.NOT_CONTAINS
        UacFeedFilterOperator.EXISTS -> Operator.EXISTS
        UacFeedFilterOperator.UNKNOWN -> Operator.UNKNOWN
    }

    private fun Operator.toUacFeedFilterOperator(): UacFeedFilterOperator = when(this) {
        Operator.GREATER -> UacFeedFilterOperator.GREATER
        Operator.LESS -> UacFeedFilterOperator.LESS
        Operator.EQUALS -> UacFeedFilterOperator.EQUALS
        Operator.NOT_EQUALS -> UacFeedFilterOperator.UNKNOWN
        Operator.RANGE -> UacFeedFilterOperator.RANGE
        Operator.CONTAINS -> UacFeedFilterOperator.CONTAINS
        Operator.NOT_CONTAINS -> UacFeedFilterOperator.NOT_CONTAINS
        Operator.EXISTS -> UacFeedFilterOperator.EXISTS
        Operator.UNKNOWN -> UacFeedFilterOperator.UNKNOWN
    }

    fun List<String>.toFeedFilterValue(): String = JsonUtils.toJson(this)

    fun String.toFeedFilterValues(): List<String> = JsonUtils.fromJson(this,
        object : TypeReference<List<String>>() {}
    )

    fun defaultDynamicFilter(groupId: Long, name: String, feed: Feed): DynamicFeedAdTarget = DynamicFeedAdTarget()
            .withAdGroupId(groupId)
            .withConditionName(name)
            .withTab(DynamicAdTargetTab.ALL_PRODUCTS)
            .withIsSuspended(false)
            .withBusinessType(feed.businessType)
            .withFeedType(feed.feedType)
            .withCondition(listOf())
            .withConditionHash(DynamicTextAdTargetHashUtils.getHashForDynamicFeedRules(listOf()))

    fun List<UacFeedFilter>.toSmartFilters(
            groupId: Long,
            name: String,
            feed: Feed,
            filterSchema: FilterSchema): List<PerformanceFilter> {
        return this.map { it.toSmartFilter(groupId, name, feed, filterSchema) }
    }

    private fun UacFeedFilter.toSmartFilter(
            groupId: Long,
            name: String,
            feed: Feed,
            filterSchema: FilterSchema): PerformanceFilter {
        return PerformanceFilter()
                .withPid(groupId)
                .withName(name)
                .withTargetFunnel(TargetFunnel.SAME_PRODUCTS)
                .withIsSuspended(false)
                .withBusinessType(feed.businessType)
                .withFeedType(feed.feedType)
                .withSource(feed.source)
                .withTab(PerformanceFilterTab.ALL_PRODUCTS)
                .withConditions(this.conditions.toSmartConditions(filterSchema))
    }

    fun List<UacFeedFilterCondition>.toSmartConditions(filterSchema: FilterSchema): List<PerformanceFilterCondition<Any>> {
        val conditions = this.map { it.toSmartCondition() }
        PerformanceFilterConditionDBFormatParser.setParsedValue(filterSchema, conditions)
        return conditions
    }

    private fun UacFeedFilterCondition.toSmartCondition() =
            PerformanceFilterCondition<Any>(this.field, this.operator.toOperator(), this.values?.toFeedFilterValue() ?: this.value)

    fun PerformanceFilter.toUacFeedFiltersConditions() = this.conditions.map {
            UacFeedFilterCondition(it.fieldName, it.operator.toUacFeedFilterOperator(), it.stringValue, it.stringValue.toFeedFilterValues())
        }.toSet()

    fun defaultSmartFilter(groupId: Long, name: String, feed: Feed): PerformanceFilter = PerformanceFilter()
            .withPid(groupId)
            .withName(name)
            .withTargetFunnel(TargetFunnel.SAME_PRODUCTS)
            .withIsSuspended(false)
            .withBusinessType(feed.businessType)
            .withFeedType(feed.feedType)
            .withSource(feed.source)
            .withTab(PerformanceFilterTab.ALL_PRODUCTS)
            .withConditions(listOf())

    fun List<UacFeedFilter>.toFeedFilter(filterSchema: FilterSchema): FeedFilter {
        val conditions = this.flatMap { it.conditions.toFeedFilterConditions(filterSchema) }
        return FeedFilter()
            .withTab(FeedFilterTab.ALL_PRODUCTS)
            .withConditions(conditions)
    }

    private fun List<UacFeedFilterCondition>.toFeedFilterConditions(filterSchema: FilterSchema): List<FeedFilterCondition<Any>> {
        val conditions = this.map { it.toFeedFilterCondition() }
        PerformanceFilterConditionDBFormatParser.setParsedValue(filterSchema, conditions)
        return conditions
    }

    private fun UacFeedFilterCondition.toFeedFilterCondition() =
        FeedFilterCondition<Any>(this.field, this.operator.toOperator(), this.values?.toFeedFilterValue() ?: this.value)

    fun defaultFeedFilter(): FeedFilter = FeedFilter()
        .withTab(FeedFilterTab.ALL_PRODUCTS)
        .withConditions(listOf())

    fun defaultAvailablePerformanceCondition(): PerformanceFilterCondition<Any> =
        PerformanceFilterCondition<Any>(
            "available",
            Operator.NOT_EQUALS,
            "false")
            .withParsedValue(false)

    fun defaultAvailableDynamicFeedRule(): DynamicFeedRule<Any> =
        DynamicFeedRule<Any>(
            "available",
            Operator.NOT_EQUALS,
            "false")
            .withParsedValue(false)

    fun defaultAvailableFeedFilterCondition(): FeedFilterCondition<Any?> =
        FeedFilterCondition<Any?>(
            "available",
            Operator.NOT_EQUALS,
            "false")
            .withParsedValue(false)
}
