package ru.yandex.direct.core.entity.uac.converter

import ru.yandex.direct.core.entity.uac.model.Html5Asset
import ru.yandex.direct.core.entity.uac.model.ImageAsset
import ru.yandex.direct.core.entity.uac.model.MdsInfo
import ru.yandex.direct.core.entity.uac.model.MediaAsset
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.Sitelink
import ru.yandex.direct.core.entity.uac.model.SitelinkAsset
import ru.yandex.direct.core.entity.uac.model.VideoAsset
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.grut.api.AssetGrut
import ru.yandex.direct.core.grut.api.Html5AssetGrut
import ru.yandex.direct.core.grut.api.ImageAssetGrut
import ru.yandex.direct.core.grut.api.VideoAssetGrut
import ru.yandex.direct.utils.JsonUtils.toJson
import ru.yandex.direct.utils.fromJson
import ru.yandex.grut.objects.proto.Asset
import ru.yandex.grut.objects.proto.MediaType.EMediaType
import ru.yandex.grut.objects.proto.client.Schema

object UacGrutAssetsConverter {

    fun toSitelink(sitelink: SitelinkAsset): Sitelink {
        return Sitelink(
            title = sitelink.title,
            href = sitelink.href,
            description = sitelink.description,
        )
    }

    fun EMediaType.toMediaType(): MediaType? {
        return when (this) {
            EMediaType.MT_IMAGE -> MediaType.IMAGE
            EMediaType.MT_VIDEO -> MediaType.VIDEO
            EMediaType.MT_TEXT -> MediaType.TEXT
            EMediaType.MT_TITLE -> MediaType.TITLE
            EMediaType.MT_SITELINK -> MediaType.SITELINK
            EMediaType.MT_HTML5 -> MediaType.HTML5
            else -> null
        }
    }

    fun MediaType?.toGrutMediaType(): EMediaType {
        return when (this) {
            MediaType.IMAGE -> EMediaType.MT_IMAGE
            MediaType.VIDEO -> EMediaType.MT_VIDEO
            MediaType.TEXT -> EMediaType.MT_TEXT
            MediaType.TITLE -> EMediaType.MT_TITLE
            MediaType.SITELINK -> EMediaType.MT_SITELINK
            MediaType.HTML5 -> EMediaType.MT_HTML5
            else -> EMediaType.MT_NOT_SPECIFIED
        }
    }

    fun Asset.TMdsInfo.toMdsInfo(): MdsInfo {
        return MdsInfo(
            thumb = thumb,
            sourceUrl = sourceUrl,
            mdsUrl = if (hasMdsUrl()) mdsUrl else null,
            filename = if (hasFilename()) filename else null,
            meta = fromJson(meta),
        )
    }

    fun Schema.TAsset.toVideoAsset(): VideoAsset {
        return VideoAsset(
            id = meta.id.toIdString(),
            clientId = meta.clientId.toIdString(),
            mdsInfo = spec.video.mdsInfo.toMdsInfo(),
            videoDuration = spec.video.videoDuration,
        )
    }

    fun Schema.TAsset.toImageAsset(): ImageAsset {
        return ImageAsset(
            id = meta.id.toIdString(),
            clientId = meta.clientId.toIdString(),
            mdsInfo = spec.image.mdsInfo.toMdsInfo(),
            directImageHash = spec.image.directImageHash,
        )
    }

    fun Schema.TAsset.toSitelinkAsset(): SitelinkAsset {
        return SitelinkAsset(
            id = meta.id.toIdString(),
            clientId = meta.clientId.toIdString(),
            title = spec.sitelink.title,
            href = spec.sitelink.href,
            description = if (spec.sitelink.hasDescription()) spec.sitelink.description else null,
        )
    }

    fun Schema.TAsset.toHtml5Asset(): Html5Asset {
        return Html5Asset(
            id = meta.id.toIdString(),
            clientId = meta.clientId.toIdString(),
            mdsInfo = spec.html5.mdsInfo.toMdsInfo(),
        )
    }

    fun toAssetGrut(content: UacYdbContent): AssetGrut {
        require(content.type == MediaType.IMAGE || content.type == MediaType.VIDEO || content.type == MediaType.HTML5)
        val mdsInfo = Asset.TMdsInfo.newBuilder().apply {
            thumb = content.thumb
            sourceUrl = content.sourceUrl
            content.mdsUrl?.let { mdsUrl = it }
            content.filename?.let { filename = it }
            meta = toJson(content.meta)
        }.build()
        if (content.type == MediaType.IMAGE) {
            return ImageAssetGrut(
                id = content.id.toIdLong(),
                clientId = content.accountId!!.toIdLong(),
                mdsInfo = mdsInfo,
                imageHash = content.directImageHash
            )
        }

        if (content.type == MediaType.VIDEO) {
            return VideoAssetGrut(
                id = content.id.toIdLong(),
                clientId = content.accountId!!.toIdLong(),
                mdsInfo = mdsInfo,
                duration = content.videoDuration
            )
        }
        return Html5AssetGrut(
            id = content.id.toIdLong(),
            clientId = content.accountId!!.toIdLong(),
            mdsInfo = mdsInfo,
        )
    }

    fun toUacYdbContent(asset: Schema.TAsset): UacYdbContent? {
        return when {
            asset.spec.hasVideo() -> {
                toUacYdbContent(asset.toVideoAsset())
            }
            asset.spec.hasImage() -> {
                toUacYdbContent(asset.toImageAsset())
            }
            asset.spec.hasHtml5() -> {
                toUacYdbContent(asset.toHtml5Asset())
            }
            else -> null
        }
    }

    fun toUacYdbContent(asset: MediaAsset): UacYdbContent {
        val directImageHash: String?
        val mediaType: MediaType
        val videoDuration: Int?

        when (asset) {
            is VideoAsset -> {
                mediaType = MediaType.VIDEO
                directImageHash = null
                videoDuration = asset.videoDuration
            }
            is ImageAsset -> {
                mediaType = MediaType.IMAGE
                directImageHash = asset.directImageHash
                videoDuration = null
            }
            is Html5Asset -> {
                mediaType = MediaType.HTML5
                directImageHash = null
                videoDuration = null
            }
            else -> throw IllegalArgumentException("Unsupported asset type: \"${asset.javaClass.simpleName}\"")
        }
        return UacYdbContent(
            id = asset.id,
            ownerId = asset.clientId,
            type = mediaType,
            thumb = asset.mdsInfo.thumb,
            sourceUrl = asset.mdsInfo.sourceUrl,
            mdsUrl = asset.mdsInfo.mdsUrl,
            meta = asset.mdsInfo.meta,
            videoDuration = videoDuration,
            filename = asset.mdsInfo.filename,
            accountId = asset.clientId,
            directImageHash = directImageHash,
        )
    }
}
