package ru.yandex.direct.core.entity.uac.converter

import com.google.protobuf.Descriptors
import com.google.protobuf.ProtocolMessageEnum
import java.time.LocalDateTime
import java.time.LocalTime
import java.time.ZoneOffset
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.ConcurrentMap
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagData
import ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.toFeedFilterValue
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.toFeedFilterValues
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.AssetLinkType
import ru.yandex.direct.core.entity.uac.model.CampaignContentStatus
import ru.yandex.direct.core.entity.uac.model.DeviceType
import ru.yandex.direct.core.entity.uac.model.DirectCampaignStatus
import ru.yandex.direct.core.entity.uac.model.Gender
import ru.yandex.direct.core.entity.uac.model.HolidaySettings
import ru.yandex.direct.core.entity.uac.model.InventoryType
import ru.yandex.direct.core.entity.uac.model.LimitPeriodType
import ru.yandex.direct.core.entity.uac.model.Socdem
import ru.yandex.direct.core.entity.uac.model.TargetStatus
import ru.yandex.direct.core.entity.uac.model.TargetType
import ru.yandex.direct.core.entity.uac.model.TimeTarget
import ru.yandex.direct.core.entity.uac.model.UacAdjustment
import ru.yandex.direct.core.entity.uac.model.UacCampaignOptions
import ru.yandex.direct.core.entity.uac.model.UacFeedFilter
import ru.yandex.direct.core.entity.uac.model.UacFeedFilterCondition
import ru.yandex.direct.core.entity.uac.model.UacFeedFilterOperator
import ru.yandex.direct.core.entity.uac.model.UacGoal
import ru.yandex.direct.core.entity.uac.model.UacGoalType
import ru.yandex.direct.core.entity.uac.model.UacRetargetingCondition
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRule
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoal
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoalType
import ru.yandex.direct.core.entity.uac.model.UacShowsFrequencyLimit
import ru.yandex.direct.core.entity.uac.model.UacStrategy
import ru.yandex.direct.core.entity.uac.model.UacStrategyData
import ru.yandex.direct.core.entity.uac.model.UacStrategyName
import ru.yandex.direct.core.entity.uac.model.UacStrategyPlatform
import ru.yandex.direct.core.entity.uac.model.relevance_match.UacRelevanceMatch
import ru.yandex.direct.core.entity.uac.model.relevance_match.UacRelevanceMatchCategory
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.moneyFromDb
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.moneyToDb
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.CpmAssetButton
import ru.yandex.direct.core.entity.uac.repository.ydb.model.CpmAssetLogo
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacBannerMeasurerSystem
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacBrandsafety
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacButtonAction
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacCampaignMeasurer
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacCampaignMeasurerSystem
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacCpmAsset
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacDisabledPlaces
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacMeasurer
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacSearchLift
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.grut.objects.proto.AgePoint.EAgePoint
import ru.yandex.grut.objects.proto.AssetLink
import ru.yandex.grut.objects.proto.AssetLink.TAssetLinks
import ru.yandex.grut.objects.proto.AssetLink.TAssetLinksStatuses
import ru.yandex.grut.objects.proto.Campaign.ECampaignTypeOld
import ru.yandex.grut.objects.proto.Campaign.TBriefStrategy
import ru.yandex.grut.objects.proto.Campaign.TBriefStrategy.ECampaignPlatform
import ru.yandex.grut.objects.proto.Campaign.TBriefStrategy.EGoalType
import ru.yandex.grut.objects.proto.Campaign.TBriefStrategy.ELimitPeriod
import ru.yandex.grut.objects.proto.Campaign.TBriefStrategy.EMobileAppCampaignTargetType
import ru.yandex.grut.objects.proto.Campaign.TBriefStrategy.EStrategyName
import ru.yandex.grut.objects.proto.Campaign.TBriefStrategy.TGoal
import ru.yandex.grut.objects.proto.Campaign.TBriefStrategy.TStrategyData
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.EBriefStatus
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.ECryptaInterestType
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.EDeviceType
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.EFeedFilterOperator
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.TAdjustment
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.TCpmAsset
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.TCpmData
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.TRetargetingCondition
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.TRetargetingConditionRule
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.TRetargetingConditionRule.ERetargetingConditionRuleType
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.TRetargetingConditionRule.TRetargetingConditionRuleGoal.ERetargetingConditionRuleGoalType
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.TShowsFrequencyLimit
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.TSocdem
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.TTimeTarget
import ru.yandex.grut.objects.proto.Campaign.TCampaignData
import ru.yandex.grut.objects.proto.Campaign.TCampaignSpec
import ru.yandex.grut.objects.proto.ContentFlags
import ru.yandex.grut.objects.proto.InventoryType.EInventoryType
import ru.yandex.grut.objects.proto.RejectReasons
import ru.yandex.grut.objects.proto.client.Schema.TCampaign

object UacGrutCampaignConverter {
    private const val AGE_ENUM_NAME_PREFIX = "age_"
    private const val BABY_FOOD_ENUM_NAME_PREFIX = "baby_food"

    fun toCampaignSpec(campaign: UacYdbCampaign): TCampaignSpec {
        return TCampaignSpec.newBuilder().apply {
            campaignBrief = TCampaignBrief.newBuilder().apply {
                targetHref = buildTargetHref(campaign)
                campaign.appId?.let { appId = it }
                campaign.regions?.let { addAllRegions(it) }
                campaign.minusRegions?.let { addAllMinusRegions(it) }
                strategy = buildCampaignStrategy(campaign)
                targetStatus = campaign.targetStatus.toEBriefStatus()
                campaign.skadNetworkEnabled?.let { skadNetworkEnabled = it }
                campaign.adultContentEnabled?.let { adultContentEnabled = it }
                campaign.socdem?.let {
                    socdem = buildSocdem(it)
                }
                campaign.hyperGeoId?.let { hypergeoId = it }
                campaign.permalinkId?.let { permalinkId = it }
                campaign.phoneId?.let { phoneId = it }
                campaign.keywords?.let { addAllKeywords(it) }
                campaign.minusKeywords?.let { addAllMinusKeywords(it) }
                campaign.deviceTypes?.let { addAllDeviceType(it.toEDeviceTypes()) }
                campaign.videosAreNonSkippable?.let { videosAreNonSkippable = it }
                campaign.calltrackingSettingsId?.let { calltrackingSettingsId = it }
                campaign.briefSynced?.let { briefSynced = it }
                campaign.timeTarget?.let { timeTarget = toTTimeTarget(campaign.timeTarget) }
                campaign.retargetingCondition?.let { retargetingCondition = buildRetargetingCondition(it) }
                campaign.contentFlags?.let { contentFlags = contentFlagsMapToGrutContentFlags(it) }
                buildEcomData(campaign)?.let { ecom = it }
                buildCpmData(campaign)?.let { cpmData = it }
                campaign.zenPublisherId?.let { zenPublisherId = it }
                campaign.adjustments?.let { addAllAdjustments(it.toTAdjustments()) }
                campaign.trackingParams?.let { trackingParams = it }
                campaign.recommendationsManagementEnabled?.let { recommendationsManagementEnabled = it }
                campaign.priceRecommendationsManagementEnabled?.let { priceRecommendationsManagementEnabled = it }
                campaign.inventoryTypes?.let { addAllInventoryType(it.toEInventoryTypes()) }
                campaign.relevanceMatch?.let { relevanceMatch = toTRelevanceMatch(it) }
                campaign.uacDisabledPlaces?.let { disabledPlaces = buildDisabledPlaces(it) }
                campaign.showTitleAndBody?.let { showTitleAndBody = it }
                campaign.audienceSegmentsSynchronized?.let { audienceSegmentsSynchronized = it }
            }.build()
            campaignData = TCampaignData.newBuilder().apply {
                name = campaign.name
                campaign.phoneId?.let { phoneId = it }
                campaign.counters?.let { addAllCounters(it) }
            }.build()
            briefAssetLinks = TAssetLinks.newBuilder().apply {
                campaign.assetLinks?.let { addAllLinks(it.toAssetLinks()) }
            }.build()
            briefAssetLinksStatuses = TAssetLinksStatuses.newBuilder().apply {
                campaign.assetLinks?.let { addAllLinkStatuses(it.toAssetLinkStatuses()) }
            }.build()
            updateTime = UacYdbUtils.toEpochSecond(campaign.updatedAt).toInt()
            campaign.startedAt?.let { startTime = UacYdbUtils.toEpochSecond(it).toInt() }
            campaign.directCampaignStatus?.let { status = it.toECampaignStatus() }
            campaign.bizLandingId?.let { bizLandingId = it }
        }.build()
    }

    fun TCampaign.toUacYdbCampaign(): UacYdbCampaign {
        return getUacCampaign(this)
    }

    fun TCampaign.toUacYdbCampaignHistory(campaign: UacYdbCampaign): UacYdbCampaign {
        val brief = spec.campaignBrief
        val advType = if (meta.hasCampaignType()) meta.campaignType.toAdvType()!! else campaign.advType
        val targetStatus = if (brief.hasTargetStatus()) {
            brief.targetStatus.toTargetStatus()!!
        } else {
            campaign.targetStatus
        }
        return getUacCampaign(
            this,
            advType,
            if (meta.hasId()) meta.id.toIdString() else campaign.id,
            if (meta.hasClientId()) meta.clientId.toIdString() else campaign.account,
            targetStatus,
        )
    }

    private fun getDefaultInventoryTypes(type: AdvType): Set<InventoryType>? {
        if (type == AdvType.CPM_BANNER) {
            return setOf(InventoryType.INAPP, InventoryType.INPAGE, InventoryType.INSTREAM, InventoryType.REWARDED)
        }
        return null
    }

    fun getDefaultDeviceTypes(isCpm: Boolean): Set<DeviceType>? {
        if (isCpm) {
            return setOf(DeviceType.DESKTOP, DeviceType.PHONE, DeviceType.SMART_TV)
        }
        return null
    }

    fun buildSocdem(
        socdem: Socdem,
    ): TSocdem {
        return TSocdem.newBuilder().apply {
            ageLower = socdem.ageLower.toEAgePoint()
            ageUpper = socdem.ageUpper.toEAgePoint()
            addAllGenders(socdem.genders.toEGenders())
            socdem.incomeLower?.let { incomeLower = it.toEIncomeGrade() }
            socdem.incomeUpper?.let { incomeUpper = it.toEIncomeGrade() }
        }.build()
    }

    fun readAssetLinks(
        assetLinks: TAssetLinks,
        assetLinksStatuses: TAssetLinksStatuses,
        campaignId: String
    ): List<UacYdbCampaignContent> {
        if (assetLinks.linksCount == 0) {
            return emptyList()
        }
        val linksStatusesMap = assetLinksStatuses.linkStatusesList
            .associateBy { Pair(if (it.hasAssetLinkId()) it.assetLinkId else it.assetId, it.linkType) }
        return assetLinks.linksList.map { assetLink ->
            val assetLinkId = if (assetLink.hasId()) assetLink.id else assetLink.assetId
            val assetLinkStatus = linksStatusesMap[Pair(assetLinkId, assetLink.linkType)]

            UacYdbCampaignContent(
                id = assetLinkId.toIdString(),
                campaignId = campaignId,
                contentId = assetLink.assetId.toIdString(),
                type = null,
                order = assetLink.order,
                createdAt = DateTimeUtils.fromEpochSeconds(assetLink.createTime.toLong()),
                removedAt = if (assetLink.hasRemoveTime()) DateTimeUtils.fromEpochSeconds(assetLink.removeTime.toLong()) else null,
                status = assetLinkStatus?.status?.toCampaignContentStatus() ?: CampaignContentStatus.CREATED,
                rejectReasons = if (assetLinkStatus != null && assetLinkStatus.hasRejectReasons()) {
                    assetLinkStatus.rejectReasons.rejectReasonsList.map { it.toModerationDiagData() }
                } else {
                    null
                },
                linkType = if (assetLink.hasLinkType()) assetLink.linkType.toAssetLinkType() else null,
            )
        }
    }

    fun List<UacYdbCampaignContent>.toAssetLinks(): List<AssetLink.TAssetLink> = map { link ->
        AssetLink.TAssetLink.newBuilder().apply {
            id = link.id.toIdLong()
            link.contentId?.let { assetId = it.toIdLong() }
            link.linkType?.let { linkType = it.toELinkType() }
            createTime = UacYdbUtils.toEpochSecond(link.createdAt).toInt()
            link.removedAt?.let { removeTime = UacYdbUtils.toEpochSecond(it).toInt() }
            order = link.order
        }.build()
    }

    fun List<UacYdbCampaignContent>.toAssetLinkStatuses(): List<AssetLink.TAssetLinkStatus> = map { link ->
        AssetLink.TAssetLinkStatus.newBuilder().apply {
            assetLinkId = link.id.toIdLong()
            link.contentId?.let { assetId = it.toIdLong() }
            link.linkType?.let { linkType = link.linkType.toELinkType() }
            status = link.status.toEAssetLinkStatus()
            link.rejectReasons?.let { rejectReasons = it.toRejectReasons() }
        }.build()
    }

    fun LimitPeriodType.toELimitPeriod(): ELimitPeriod {
        return when (this) {
            LimitPeriodType.WEEK -> ELimitPeriod.LP_WEEK
            LimitPeriodType.MONTH -> ELimitPeriod.LP_MONTH
        }
    }

    private fun buildTargetHref(
        campaign: UacYdbCampaign,
    ): TCampaignBrief.TTargetHref {
        return TCampaignBrief.TTargetHref.newBuilder().apply {
            href = campaign.storeUrl
            campaign.trackingUrl?.let { trackingUrl = it }
            campaign.impressionUrl?.let { impressionUrl = it }
            // TODO: разобраться с turbolanding id
        }.build()
    }

    private fun buildCampaignStrategy(
        campaign: UacYdbCampaign,
    ): TBriefStrategy {
        return TBriefStrategy.newBuilder().apply {
            campaign.options?.limitPeriod?.let { limitPeriod = it.toELimitPeriod() }
            campaign.weekLimit?.let { weekLimit = moneyToDb(it) }
            campaign.cpa?.let { cpa = moneyToDb(it) }
            campaign.crr?.let { crr = it }
            campaign.targetId?.let { mobileAppCampaignTargetType = it.toEMobileAppCampaignTargetType() }
            campaign.goals?.let {
                addAllGoals(it.toTGoals())
            }
            campaign.strategy?.let { strategy ->
                strategyName = strategy.uacStrategyName.toEStrategyName()
                strategyData = buildStrategyData(strategy.uacStrategyData)
            }
            campaign.strategyPlatform?.let { platform = it.toECampaignPlatform() }
        }.build()
    }

    fun TargetType.toEMobileAppCampaignTargetType(): EMobileAppCampaignTargetType {
        return when (this) {
            TargetType.INSTALL -> EMobileAppCampaignTargetType.MACTT_INSTALL
            TargetType.GOAL -> EMobileAppCampaignTargetType.MACTT_GOAL
            TargetType.LAUNCH -> EMobileAppCampaignTargetType.MACTT_LAUNCH
            TargetType.PAYMENT -> EMobileAppCampaignTargetType.MACTT_PAYMENT
            TargetType.BASKET -> EMobileAppCampaignTargetType.MACTT_BASKET
            TargetType.WISH_LIST -> EMobileAppCampaignTargetType.MACTT_WISH_LIST
            TargetType.REGISTER -> EMobileAppCampaignTargetType.MACTT_REGISTER
            TargetType.TRAIN -> EMobileAppCampaignTargetType.MACTT_TRAIN
            TargetType.ORDER -> EMobileAppCampaignTargetType.MACTT_ORDER
            TargetType.BUY -> EMobileAppCampaignTargetType.MACTT_BUY
            TargetType.RATING -> EMobileAppCampaignTargetType.MACTT_RATING
            TargetType.SEARCH -> EMobileAppCampaignTargetType.MACTT_SEARCH
            TargetType.CREDITS -> EMobileAppCampaignTargetType.MACTT_CREDITS
            TargetType.LEVELUP -> EMobileAppCampaignTargetType.MACTT_LEVELUP
            TargetType.WATCH -> EMobileAppCampaignTargetType.MACTT_WATCH
            TargetType.TIMESPENT -> EMobileAppCampaignTargetType.MACTT_TIMESPENT
            TargetType.SHARE -> EMobileAppCampaignTargetType.MACTT_SHARE
            TargetType.USER_1 -> EMobileAppCampaignTargetType.MACTT_USER_1
            TargetType.USER_2 -> EMobileAppCampaignTargetType.MACTT_USER_2
            TargetType.USER_3 -> EMobileAppCampaignTargetType.MACTT_USER_3
            TargetType.CPC -> EMobileAppCampaignTargetType.MACTT_CPC
        }
    }

    fun ECampaignTypeOld.toAdvType(): AdvType? {
        return when (this) {
            ECampaignTypeOld.CTO_TEXT -> AdvType.TEXT
            ECampaignTypeOld.CTO_MOBILE_APP -> AdvType.MOBILE_CONTENT
            ECampaignTypeOld.CTO_CPM_BANNER -> AdvType.CPM_BANNER
            else -> return null
        }
    }

    fun List<UacGoal>.toTGoals(): List<TGoal> {
        return map {
            TGoal.newBuilder().apply {
                goalId = it.goalId
                it.goalType?.let { goalType = it.toEGoalType() }
                it.conversionValue?.let { conversionValue = moneyToDb(it) }
                it.cpa?.let{ cpa = moneyToDb(it) }
            }.build()
        }
    }

    fun List<TGoal>.toUacGoals(): List<UacGoal> {
        return map {
            UacGoal(
                it.goalId,
                if (it.hasGoalType()) it.goalType.toUacGoalType() else null,
                if (it.hasConversionValue()) moneyFromDb(it.conversionValue) else null,
                if (it.hasCpa()) moneyFromDb(it.cpa) else null
            )
        }
    }

    fun UacGoalType.toEGoalType(): EGoalType {
        return when (this) {
            UacGoalType.ECOMMERCE -> EGoalType.GT_ECOMMERCE
            UacGoalType.CALL -> EGoalType.GT_CALL
            UacGoalType.OFFLINE -> EGoalType.GT_OFFLINE
            UacGoalType.OTHER -> EGoalType.GT_UNKNOWN
        }
    }

    fun EGoalType.toUacGoalType(): UacGoalType {
        return when (this) {
            EGoalType.GT_UNKNOWN -> UacGoalType.OTHER
            EGoalType.GT_ECOMMERCE -> UacGoalType.ECOMMERCE
            EGoalType.GT_CALL -> UacGoalType.CALL
            EGoalType.GT_OFFLINE -> UacGoalType.OFFLINE
        }
    }

    fun ru.yandex.direct.core.entity.uac.model.AgePoint.toEAgePoint(): EAgePoint {
        return when (this) {
            ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_0 -> EAgePoint.AP_AGE_0
            ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_18 -> EAgePoint.AP_AGE_18
            ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_25 -> EAgePoint.AP_AGE_25
            ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_35 -> EAgePoint.AP_AGE_35
            ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_45 -> EAgePoint.AP_AGE_45
            ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_55 -> EAgePoint.AP_AGE_55
            ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_INF -> EAgePoint.AP_AGE_INF
        }
    }

    fun EAgePoint.toAgePoint(): ru.yandex.direct.core.entity.uac.model.AgePoint? {
        return when (this) {
            EAgePoint.AP_AGE_0 -> ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_0
            EAgePoint.AP_AGE_18 -> ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_18
            EAgePoint.AP_AGE_25 -> ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_25
            EAgePoint.AP_AGE_35 -> ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_35
            EAgePoint.AP_AGE_45 -> ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_45
            EAgePoint.AP_AGE_55 -> ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_55
            EAgePoint.AP_AGE_INF -> ru.yandex.direct.core.entity.uac.model.AgePoint.AGE_INF
            else -> null
        }
    }

    fun ELimitPeriod.toLimitPeriodType(): LimitPeriodType? {
        return when (this) {
            ELimitPeriod.LP_WEEK -> LimitPeriodType.WEEK
            ELimitPeriod.LP_MONTH -> LimitPeriodType.MONTH
            else -> null
        }
    }

    fun List<Gender>.toEGenders(): List<ru.yandex.grut.objects.proto.Gender.EGender> = map { it.toEGender() }

    fun List<ru.yandex.grut.objects.proto.Gender.EGender>.toGenders(): List<Gender> = map { it.toGender()!! }

    fun Gender.toEGender(): ru.yandex.grut.objects.proto.Gender.EGender {
        return when (this) {
            Gender.MALE -> ru.yandex.grut.objects.proto.Gender.EGender.G_MALE
            Gender.FEMALE -> ru.yandex.grut.objects.proto.Gender.EGender.G_FEMALE
        }
    }

    private fun ru.yandex.grut.objects.proto.Gender.EGender.toGender(): Gender? {
        return when (this) {
            ru.yandex.grut.objects.proto.Gender.EGender.G_MALE -> Gender.MALE
            ru.yandex.grut.objects.proto.Gender.EGender.G_FEMALE -> Gender.FEMALE
            else -> null
        }
    }

    fun EMobileAppCampaignTargetType.toTargetType(): TargetType? {
        return when (this) {
            EMobileAppCampaignTargetType.MACTT_INSTALL -> TargetType.INSTALL
            EMobileAppCampaignTargetType.MACTT_GOAL -> TargetType.GOAL
            EMobileAppCampaignTargetType.MACTT_LAUNCH -> TargetType.LAUNCH
            EMobileAppCampaignTargetType.MACTT_PAYMENT -> TargetType.PAYMENT
            EMobileAppCampaignTargetType.MACTT_BASKET -> TargetType.BASKET
            EMobileAppCampaignTargetType.MACTT_WISH_LIST -> TargetType.WISH_LIST
            EMobileAppCampaignTargetType.MACTT_REGISTER -> TargetType.REGISTER
            EMobileAppCampaignTargetType.MACTT_TRAIN -> TargetType.TRAIN
            EMobileAppCampaignTargetType.MACTT_ORDER -> TargetType.ORDER
            EMobileAppCampaignTargetType.MACTT_BUY -> TargetType.BUY
            EMobileAppCampaignTargetType.MACTT_RATING -> TargetType.RATING
            EMobileAppCampaignTargetType.MACTT_SEARCH -> TargetType.SEARCH
            EMobileAppCampaignTargetType.MACTT_CREDITS -> TargetType.CREDITS
            EMobileAppCampaignTargetType.MACTT_LEVELUP -> TargetType.LEVELUP
            EMobileAppCampaignTargetType.MACTT_WATCH -> TargetType.WATCH
            EMobileAppCampaignTargetType.MACTT_TIMESPENT -> TargetType.TIMESPENT
            EMobileAppCampaignTargetType.MACTT_SHARE -> TargetType.SHARE
            EMobileAppCampaignTargetType.MACTT_USER_1 -> TargetType.USER_1
            EMobileAppCampaignTargetType.MACTT_USER_2 -> TargetType.USER_2
            EMobileAppCampaignTargetType.MACTT_USER_3 -> TargetType.USER_3
            EMobileAppCampaignTargetType.MACTT_CPC -> TargetType.CPC
            else -> null
        }
    }

    fun EBriefStatus.toTargetStatus(): TargetStatus? {
        return when (this) {
            EBriefStatus.BS_STARTED -> TargetStatus.STARTED
            EBriefStatus.BS_STOPPED -> TargetStatus.STOPPED
            else -> null
        }
    }

    fun TargetStatus.toEBriefStatus(): EBriefStatus {
        return when (this) {
            TargetStatus.STARTED -> EBriefStatus.BS_STARTED
            TargetStatus.STOPPED -> EBriefStatus.BS_STOPPED
        }
    }

    fun TSocdem.toSocdem(): Socdem {
        return Socdem(
            gendersList.toGenders(),
            ageLower.toAgePoint()!!,
            ageUpper.toAgePoint()!!,
            incomeLower.toIncomeGrade(),
            incomeUpper.toIncomeGrade()
        )
    }

    fun EDeviceType.toDeviceType(): DeviceType? {
        return when (this) {
            EDeviceType.DT_DESKTOP -> DeviceType.DESKTOP
            EDeviceType.DT_PHONE -> DeviceType.PHONE
            EDeviceType.DT_TABLET -> DeviceType.TABLET
            EDeviceType.DT_SMART_TV -> DeviceType.SMART_TV
            EDeviceType.DT_PHONE_IOS -> DeviceType.PHONE_IOS
            EDeviceType.DT_PHONE_ANDROID -> DeviceType.PHONE_ANDROID
            EDeviceType.DT_ALL -> DeviceType.ALL
            else -> null
        }
    }

    fun DeviceType.toEDeviceType(): EDeviceType {
        return when (this) {
            DeviceType.DESKTOP -> EDeviceType.DT_DESKTOP
            DeviceType.PHONE -> EDeviceType.DT_PHONE
            DeviceType.PHONE_IOS -> EDeviceType.DT_PHONE_IOS
            DeviceType.PHONE_ANDROID -> EDeviceType.DT_PHONE_ANDROID
            DeviceType.TABLET -> EDeviceType.DT_TABLET
            DeviceType.SMART_TV -> EDeviceType.DT_SMART_TV
            DeviceType.ALL -> EDeviceType.DT_ALL
        }
    }

    fun List<EDeviceType>.toDeviceTypes(): Set<DeviceType> {
        return map { it.toDeviceType()!! }.toSet()
    }

    fun Set<DeviceType>.toEDeviceTypes(): List<EDeviceType> {
        return map { it.toEDeviceType() }
    }

    fun EInventoryType.toInventoryType(): InventoryType? {
        return when (this) {
            EInventoryType.IT_INAPP -> InventoryType.INAPP
            EInventoryType.IT_INPAGE -> InventoryType.INPAGE
            EInventoryType.IT_INSTREAM -> InventoryType.INSTREAM
            EInventoryType.IT_REWARDED -> InventoryType.REWARDED
            EInventoryType.IT_ALL -> InventoryType.ALL              // не поддерживается
            EInventoryType.IT_ALL_BANNERS -> InventoryType.ALL
            EInventoryType.IT_INBANNER -> null
            EInventoryType.IT_UNKNOWN -> null
        }
    }

    fun InventoryType.toEInventoryType(): EInventoryType {
        return when (this) {
            InventoryType.INAPP -> EInventoryType.IT_INAPP
            InventoryType.INPAGE -> EInventoryType.IT_INPAGE
            InventoryType.INSTREAM -> EInventoryType.IT_INSTREAM
            InventoryType.REWARDED -> EInventoryType.IT_REWARDED
            InventoryType.ALL -> EInventoryType.IT_ALL
        }
    }

    fun List<EInventoryType>.toInventoryTypes(): Set<InventoryType> {
        return mapNotNull { it.toInventoryType() }.toSet()
    }

    fun Set<InventoryType>.toEInventoryTypes(): List<EInventoryType> {
        return map { it.toEInventoryType() }
    }

    fun TSocdem.EIncomeGrade.toIncomeGrade(): Socdem.IncomeGrade? {
        return when (this) {
            TSocdem.EIncomeGrade.IG_LOW -> Socdem.IncomeGrade.LOW
            TSocdem.EIncomeGrade.IG_MIDDLE -> Socdem.IncomeGrade.MIDDLE
            TSocdem.EIncomeGrade.IG_HIGH -> Socdem.IncomeGrade.HIGH
            TSocdem.EIncomeGrade.IG_PREMIUM -> Socdem.IncomeGrade.PREMIUM
            else -> null
        }
    }

    fun Socdem.IncomeGrade.toEIncomeGrade(): TSocdem.EIncomeGrade {
        return when (this) {
            Socdem.IncomeGrade.LOW -> TSocdem.EIncomeGrade.IG_LOW
            Socdem.IncomeGrade.MIDDLE -> TSocdem.EIncomeGrade.IG_MIDDLE
            Socdem.IncomeGrade.HIGH -> TSocdem.EIncomeGrade.IG_HIGH
            Socdem.IncomeGrade.PREMIUM -> TSocdem.EIncomeGrade.IG_PREMIUM
            Socdem.IncomeGrade.HIGHER_THAN_MIDDLE -> TSocdem.EIncomeGrade.IG_HIGHER_THAN_MIDDLE
        }
    }

    fun TTimeTarget.toTimeTarget(): TimeTarget {
        val holidaysSettings = if (hasHolidaysSettings()) {
            HolidaySettings(
                startHour = if (holidaysSettings.hasStartHour()) holidaysSettings.startHour else null,
                endHour = if (holidaysSettings.hasEndHour()) holidaysSettings.endHour else null,
                rateCorrections = if (holidaysSettings.hasRateCorrections()) holidaysSettings.rateCorrections else null,
                show = holidaysSettings.show
            )
        } else null
        return TimeTarget(
            timeBoard = timeBoardList.map { it.itemsList },
            useWorkingWeekends = useWorkingWeekends,
            holidaysSettings = holidaysSettings,
            enabledHolidaysMode = enabledHolidaysMode,
            idTimeZone = idTimeZone,
        )
    }

    private fun toTTimeTarget(timeTarget: TimeTarget): TTimeTarget {
        return TTimeTarget.newBuilder().apply {
            addAllTimeBoard(timeTarget.timeBoard.map {
                TTimeTarget.TTimeBoardItem.newBuilder().addAllItems(it).build()
            })
            useWorkingWeekends = timeTarget.useWorkingWeekends
            if (timeTarget.holidaysSettings != null) {
                holidaysSettings = TTimeTarget.THolidaySettings.newBuilder().apply {
                    if (timeTarget.holidaysSettings.startHour != null) startHour = timeTarget.holidaysSettings.startHour
                    if (timeTarget.holidaysSettings.endHour != null) endHour = timeTarget.holidaysSettings.endHour
                    if (timeTarget.holidaysSettings.rateCorrections != null) rateCorrections =
                        timeTarget.holidaysSettings.rateCorrections
                    show = timeTarget.holidaysSettings.show
                }.build()
            }
            enabledHolidaysMode = timeTarget.enabledHolidaysMode
            idTimeZone = timeTarget.idTimeZone
        }.build()
    }

    fun AssetLink.TAssetLinkStatus.EAssetLinkStatus.toCampaignContentStatus(): CampaignContentStatus? {
        return when (this) {
            AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_CREATED -> CampaignContentStatus.CREATED
            AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_REJECTED -> CampaignContentStatus.REJECTED
            AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_MODERATING -> CampaignContentStatus.MODERATING
            AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_ACTIVE -> CampaignContentStatus.ACTIVE
            AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_DELETED -> CampaignContentStatus.DELETED
            else -> null
        }
    }

    fun CampaignContentStatus.toEAssetLinkStatus(): AssetLink.TAssetLinkStatus.EAssetLinkStatus {
        return when (this) {
            CampaignContentStatus.CREATED -> AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_CREATED
            CampaignContentStatus.REJECTED -> AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_REJECTED
            CampaignContentStatus.MODERATING -> AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_MODERATING
            CampaignContentStatus.ACTIVE -> AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_ACTIVE
            CampaignContentStatus.DELETED -> AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_DELETED
        }
    }

    fun AssetLink.ELinkType.toAssetLinkType(): AssetLinkType {
        return when (this) {
            AssetLink.ELinkType.LT_NOT_SPECIFIED -> AssetLinkType.NOT_SPECIFED
        }
    }

    fun AssetLinkType.toELinkType(): AssetLink.ELinkType {
        return when (this) {
            AssetLinkType.NOT_SPECIFED -> AssetLink.ELinkType.LT_NOT_SPECIFIED
        }
    }

    fun RejectReasons.TRejectReason.toModerationDiagData(): ModerationDiagData {
        // TODO использовать https://a.yandex-team.ru/arc_vcs/direct/core/src/main/java/ru/yandex/direct/core/entity/moderationdiag/ModerationDiagDataConverter.kt?rev=35205951925edcb798048d42b53b7069cbb47a21#L21 ?
        fun booleanToString(b: Boolean?): String {
            return if (b == true) {
                "Yes"
            } else {
                return if (b == false) "No" else ""
            }
        }
        return ModerationDiagData(
            showDetailsUrl = if (showDetailsUrl) 1 else 0,
            diagId = diagId,
            badReason = booleanToString(if (hasBadReason()) badReason else null),
            unbanIsProhibited = booleanToString(if (hasUnbanIsProhibited()) unbanIsProhibited else null),
            diagText = diagText,
            token = token,
            shortText = shortText,
            allowFirstAid = booleanToString(if (hasAllowFirstAid()) allowFirstAid else null),
        )
    }

    fun List<ModerationDiagData>.toRejectReasons(): RejectReasons.TRejectReasons {
        return RejectReasons.TRejectReasons.newBuilder().apply {
            addAllRejectReasons(map { it.toRejectReason() })
        }.build()
    }

    fun ModerationDiagData.toRejectReason(): RejectReasons.TRejectReason {
        val diagData = this
        return RejectReasons.TRejectReason.newBuilder().apply {
            showDetailsUrl = diagData.showDetailsUrl > 0
            diagId = diagData.diagId
            setBooleanFromStr(diagData.badReason) { badReason = it }
            setBooleanFromStr(diagData.unbanIsProhibited) { unbanIsProhibited = it }
            diagData.diagText?.let { diagText = diagData.diagText }
            diagData.token?.let { token = diagData.token }
            diagData.shortText?.let { shortText = diagData.shortText }
            setBooleanFromStr(diagData.allowFirstAid) { allowFirstAid = it }
        }.build()
    }

    private fun setBooleanFromStr(str: String?, setter: (Boolean) -> Unit) = str?.let {
        if (it == "Yes") {
            setter(true)
        } else if (it == "No") {
            setter(false)
        }
    }

    fun AdvType.toEAdCampaignType(): ECampaignTypeOld {
        return when (this) {
            AdvType.TEXT -> ECampaignTypeOld.CTO_TEXT
            AdvType.MOBILE_CONTENT -> ECampaignTypeOld.CTO_MOBILE_APP
            AdvType.CPM_BANNER -> ECampaignTypeOld.CTO_CPM_BANNER
        }
    }

    fun ContentFlags.TContentFlags.toContentFlags(): Map<String, Any> {
        val result = mutableMapOf<String, Any>()

        if (abortion) {
            result["abortion"] = true
        }
        if (medicine) {
            result["medicine"] = true
        }
        if (medServices) {
            result["med_services"] = true
        }
        if (medEquipment) {
            result["med_equipment"] = true
        }
        if (pharmacy) {
            result["pharmacy"] = true
        }
        if (alcohol) {
            result["alcohol"] = true
        }
        if (tobacco) {
            result["tobacco"] = true
        }
        if (plus18) {
            result["plus18"] = true
        }
        if (dietarysuppl) {
            result["dietarysuppl"] = true
        }
        if (projectDeclaration) {
            result["project_declaration"] = true
        }
        if (tragic) {
            result["tragic"] = true
        }
        if (asocial) {
            result["asocial"] = true
        }
        if (unfamily) {
            result["unfamily"] = true
        }
        if (pseudoweapon) {
            result["pseudoweapon"] = true
        }
        if (forex) {
            result["forex"] = true
        }
        if (education) {
            result["education"] = true
        }
        if (people) {
            result["people"] = true
        }
        if (socialAdvertising) {
            result["social_advertising"] = true
        }
        if (notAnimated) {
            result["not_animated"] = true
        }
        if (goodface) {
            result["goodface"] = true
        }


        if (hasAge()) {
            result["age"] = getProtoEnumValueName(age).removePrefix(AGE_ENUM_NAME_PREFIX)
        }
        if (hasBabyFood()) {
            result["baby_food"] = getProtoEnumValueName(babyFood).removePrefix(BABY_FOOD_ENUM_NAME_PREFIX)
        }

        if (hasYaPages()) {
            result["ya_pages"] = getProtoEnumValueName(yaPages)
        }

        return result
    }

    fun TBriefStrategy.toUacStrategy(): UacStrategy? {
        if (this.hasStrategyName() && this.hasStrategyData()) {
            return UacStrategy(this.strategyName.toUacStrategyName()!!, this.strategyData.toUacStrategyData())
        }
        return null
    }

    fun UacStrategyName.toEStrategyName(): EStrategyName {
        return when (this) {
            UacStrategyName.CPM_DEFAULT -> EStrategyName.SN_CPM_DEFAULT
            UacStrategyName.AUTOBUDGET_AVG_CPV -> EStrategyName.SN_AUTOBUDGET_AVG_CPV
            UacStrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD -> EStrategyName.SN_AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD
            UacStrategyName.AUTOBUDGET_MAX_REACH -> EStrategyName.SN_AUTOBUDGET_MAX_REACH
            UacStrategyName.AUTOBUDGET_MAX_IMPRESSIONS -> EStrategyName.SN_AUTOBUDGET_MAX_IMPRESSIONS
            UacStrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD -> EStrategyName.SN_AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD
            UacStrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD -> EStrategyName.SN_AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD
            UacStrategyName.AUTOBUDGET_CRR -> EStrategyName.SN_AUTOBUDGET_CRR
            UacStrategyName.AUTOBUDGET_AVG_CLICK -> EStrategyName.SN_AUTOBUDGET_AVG_CLICK
            UacStrategyName.AUTOBUDGET_AVG_CPA -> EStrategyName.SN_AUTOBUDGET_AVG_CPA
            UacStrategyName.AUTOBUDGET_AVG_CPI -> EStrategyName.SN_AUTOBUDGET_AVG_CPI
        }
    }

    fun EStrategyName.toUacStrategyName(): UacStrategyName? {
        return when (this) {
            EStrategyName.SN_CPM_DEFAULT -> UacStrategyName.CPM_DEFAULT
            EStrategyName.SN_AUTOBUDGET_AVG_CPV -> UacStrategyName.AUTOBUDGET_AVG_CPV
            EStrategyName.SN_AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD -> UacStrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD
            EStrategyName.SN_AUTOBUDGET_MAX_REACH -> UacStrategyName.AUTOBUDGET_MAX_REACH
            EStrategyName.SN_AUTOBUDGET_MAX_IMPRESSIONS -> UacStrategyName.AUTOBUDGET_MAX_IMPRESSIONS
            EStrategyName.SN_AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD -> UacStrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD
            EStrategyName.SN_AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD -> UacStrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD
            EStrategyName.SN_AUTOBUDGET_CRR -> UacStrategyName.AUTOBUDGET_CRR
            EStrategyName.SN_AUTOBUDGET_AVG_CLICK -> UacStrategyName.AUTOBUDGET_AVG_CLICK
            EStrategyName.SN_AUTOBUDGET_AVG_CPA -> UacStrategyName.AUTOBUDGET_AVG_CPA
            EStrategyName.SN_AUTOBUDGET_AVG_CPI -> UacStrategyName.AUTOBUDGET_AVG_CPI
            EStrategyName.SN_NOT_SPECIFIED -> null
        }
    }

    fun DirectCampaignStatus.toECampaignStatus(): TCampaignSpec.ECampaignStatus {
        return when (this) {
            DirectCampaignStatus.CREATED -> TCampaignSpec.ECampaignStatus.CS_CREATED
            DirectCampaignStatus.DRAFT -> TCampaignSpec.ECampaignStatus.CS_DRAFT
        }
    }

    fun TCampaignSpec.ECampaignStatus.toDirectCampaignStatus(): DirectCampaignStatus? {
        return when (this) {
            TCampaignSpec.ECampaignStatus.CS_CREATED -> DirectCampaignStatus.CREATED
            TCampaignSpec.ECampaignStatus.CS_DRAFT -> DirectCampaignStatus.DRAFT
            else -> null
        }
    }

    fun buildStrategyData(
        uacStrategyData: UacStrategyData,
    ): TStrategyData {
        return TStrategyData.newBuilder().apply {
            uacStrategyData.budget?.let { budget = moneyToDb(it) }
            uacStrategyData.autoProlongation?.let { autoProlongation = it }
            uacStrategyData.avgCpm?.let { avgCpm = moneyToDb(it) }
            uacStrategyData.startDate?.let { startTime = it.toEpochSecond(LocalTime.MIN, ZoneOffset.UTC).toInt() }
            uacStrategyData.finishDate?.let { finishTime = it.toEpochSecond(LocalTime.MIN, ZoneOffset.UTC).toInt() }
            uacStrategyData.sum?.let { sum = moneyToDb(it) }
            uacStrategyData.avgCpv?.let { avgCpv = moneyToDb(it) }
            uacStrategyData.crr?.let { crr = it }
            uacStrategyData.avgCpa?.let { avgCpa = moneyToDb(it) }
            uacStrategyData.avgCpi?.let { avgCpi = moneyToDb(it) }
            uacStrategyData.avgBid?.let { avgBid = moneyToDb(it) }
            uacStrategyData.payForConversion?.let { payForConversion = it }
        }.build()
    }

    fun TStrategyData.toUacStrategyData(): UacStrategyData {
        return UacStrategyData(
            budget = if (this.hasBudget()) moneyFromDb(this.budget) else null,
            autoProlongation = this.autoProlongation,
            avgCpm = if (this.hasAvgCpm()) moneyFromDb(this.avgCpm) else null,
            startDate = LocalDateTime.ofEpochSecond(this.startTime.toLong(), 0, ZoneOffset.UTC).toLocalDate(),
            finishDate = LocalDateTime.ofEpochSecond(this.finishTime.toLong(), 0, ZoneOffset.UTC).toLocalDate(),
            sum = if (this.hasSum()) moneyFromDb(this.sum) else null,
            avgCpv = if (this.hasAvgCpv()) moneyFromDb(avgCpv) else null,
            crr = if (this.hasCrr()) crr else null,
            avgCpa = if (this.hasAvgCpa()) moneyFromDb(this.avgCpa) else null,
            avgCpi = if (this.hasAvgCpi()) moneyFromDb(this.avgCpi) else null,
            avgBid = if (this.hasAvgBid()) moneyFromDb(this.avgBid) else null,
            payForConversion = if (this.hasPayForConversion()) this.payForConversion else null,
        )
    }

    fun UacStrategyPlatform.toECampaignPlatform(): ECampaignPlatform {
        return when (this) {
            UacStrategyPlatform.SEARCH -> ECampaignPlatform.CP_SEARCH
            UacStrategyPlatform.CONTEXT -> ECampaignPlatform.CP_CONTEXT
            UacStrategyPlatform.BOTH -> ECampaignPlatform.CP_BOTH
        }
    }

    fun ECampaignPlatform.toUacStrategyPlatform(): UacStrategyPlatform {
        return when (this) {
            ECampaignPlatform.CP_SEARCH -> UacStrategyPlatform.SEARCH
            ECampaignPlatform.CP_CONTEXT -> UacStrategyPlatform.CONTEXT
            ECampaignPlatform.CP_BOTH -> UacStrategyPlatform.BOTH
            else -> UacStrategyPlatform.BOTH
        }
    }

    fun contentFlagsMapToGrutContentFlags(contentFlagsMap: Map<String, Any>): ContentFlags.TContentFlags {
        val contentFlagsBuilder = ContentFlags.TContentFlags.newBuilder()
        if (contentFlagsMap.containsKey("abortion")) {
            contentFlagsBuilder.abortion = true
        }
        if (contentFlagsMap.containsKey("medicine")) {
            contentFlagsBuilder.medicine = true
        }
        if (contentFlagsMap.containsKey("med_services")) {
            contentFlagsBuilder.medServices = true
        }
        if (contentFlagsMap.containsKey("med_equipment")) {
            contentFlagsBuilder.medEquipment = true
        }
        if (contentFlagsMap.containsKey("pharmacy")) {
            contentFlagsBuilder.pharmacy = true
        }
        if (contentFlagsMap.containsKey("alcohol")) {
            contentFlagsBuilder.alcohol = true
        }
        if (contentFlagsMap.containsKey("tobacco")) {
            contentFlagsBuilder.tobacco = true
        }
        if (contentFlagsMap.containsKey("plus18")) {
            contentFlagsBuilder.plus18 = true
        }
        if (contentFlagsMap.containsKey("dietarysuppl")) {
            contentFlagsBuilder.dietarysuppl = true
        }
        if (contentFlagsMap.containsKey("project_declaration")) {
            contentFlagsBuilder.projectDeclaration = true
        }
        if (contentFlagsMap.containsKey("tragic")) {
            contentFlagsBuilder.tragic = true
        }
        if (contentFlagsMap.containsKey("asocial")) {
            contentFlagsBuilder.asocial = true
        }
        if (contentFlagsMap.containsKey("unfamily")) {
            contentFlagsBuilder.unfamily = true
        }
        if (contentFlagsMap.containsKey("pseudoweapon")) {
            contentFlagsBuilder.pseudoweapon = true
        }
        if (contentFlagsMap.containsKey("forex")) {
            contentFlagsBuilder.forex = true
        }
        if (contentFlagsMap.containsKey("education")) {
            contentFlagsBuilder.education = true
        }
        if (contentFlagsMap.containsKey("people")) {
            contentFlagsBuilder.people = true
        }
        if (contentFlagsMap.containsKey("social_advertising")) {
            contentFlagsBuilder.socialAdvertising = true
        }
        if (contentFlagsMap.containsKey("not_animated")) {
            contentFlagsBuilder.notAnimated = true
        }
        if (contentFlagsMap.containsKey("goodface")) {
            contentFlagsBuilder.goodface = true
        }
        if (contentFlagsMap.containsKey("age")) {
            val age = ageFromString("${AGE_ENUM_NAME_PREFIX}${contentFlagsMap["age"]}")
            if (age != null) contentFlagsBuilder.age = age
        }
        if (contentFlagsMap.containsKey("baby_food")) {
            val babyFood = babyFoodFromString("${BABY_FOOD_ENUM_NAME_PREFIX}${contentFlagsMap["baby_food"]}")
            if (babyFood != null) contentFlagsBuilder.babyFood = babyFood
        }
        if (contentFlagsMap.containsKey("ya_pages")) {
            val yaPages = yaPagesFromString(contentFlagsMap["ya_pages"] as String)
            if (yaPages != null) contentFlagsBuilder.yaPages = yaPages
        }
        return contentFlagsBuilder.build()
    }

    private fun ageFromString(name: String): EAgePoint? {
        val num = PROTO_ENUM_NUMBER_HOLDER.getProtoEnumValueNumber(name, EAgePoint.getDescriptor())
        return num?.let { EAgePoint.forNumber(it) }
    }

    private fun babyFoodFromString(name: String): ContentFlags.EBabyFood? {
        val num = PROTO_ENUM_NUMBER_HOLDER.getProtoEnumValueNumber(name, ContentFlags.EBabyFood.getDescriptor())
        return num?.let { ContentFlags.EBabyFood.forNumber(it) }
    }

    private fun yaPagesFromString(name: String): ContentFlags.EYaPages? {
        val num = PROTO_ENUM_NUMBER_HOLDER.getProtoEnumValueNumber(name, ContentFlags.EYaPages.getDescriptor())
        return num?.let { ContentFlags.EYaPages.forNumber(it) }
    }

    private fun EFeedFilterOperator.toUacFeedFilterOperator(): UacFeedFilterOperator? {
        return when (this) {
            EFeedFilterOperator.FFO_GREATER -> UacFeedFilterOperator.GREATER
            EFeedFilterOperator.FFO_LESS -> UacFeedFilterOperator.LESS
            EFeedFilterOperator.FFO_EQUALS -> UacFeedFilterOperator.EQUALS
            EFeedFilterOperator.FFO_RANGE -> UacFeedFilterOperator.RANGE
            EFeedFilterOperator.FFO_CONTAINS -> UacFeedFilterOperator.CONTAINS
            EFeedFilterOperator.FFO_NOT_CONTAINS -> UacFeedFilterOperator.NOT_CONTAINS
            EFeedFilterOperator.FFO_EXISTS -> UacFeedFilterOperator.EXISTS
            EFeedFilterOperator.FFO_UNKNOWN -> UacFeedFilterOperator.UNKNOWN
            EFeedFilterOperator.FFO_NOT_SPECIFIED -> null
        }
    }

    private fun UacFeedFilterOperator.toEFeedFilterOperator(): EFeedFilterOperator {
        return when (this) {
            UacFeedFilterOperator.GREATER -> EFeedFilterOperator.FFO_GREATER
            UacFeedFilterOperator.LESS -> EFeedFilterOperator.FFO_LESS
            UacFeedFilterOperator.EQUALS -> EFeedFilterOperator.FFO_EQUALS
            UacFeedFilterOperator.RANGE -> EFeedFilterOperator.FFO_RANGE
            UacFeedFilterOperator.CONTAINS -> EFeedFilterOperator.FFO_CONTAINS
            UacFeedFilterOperator.NOT_CONTAINS -> EFeedFilterOperator.FFO_NOT_CONTAINS
            UacFeedFilterOperator.EXISTS -> EFeedFilterOperator.FFO_EXISTS
            UacFeedFilterOperator.UNKNOWN -> EFeedFilterOperator.FFO_UNKNOWN
        }
    }

    private fun ERetargetingConditionRuleType.toRetargetingConditionRuleType(): UacRetargetingConditionRule.RuleType? {
        return when (this) {
            ERetargetingConditionRuleType.RCRT_ALL -> UacRetargetingConditionRule.RuleType.ALL
            ERetargetingConditionRuleType.RCRT_NOT -> UacRetargetingConditionRule.RuleType.NOT
            ERetargetingConditionRuleType.RCRT_OR -> UacRetargetingConditionRule.RuleType.OR
            ERetargetingConditionRuleType.RCRT_NOT_SPECIFIED -> null
        }
    }

    private fun UacRetargetingConditionRule.RuleType.toERetargetingConditionRuleType(): ERetargetingConditionRuleType {
        return when (this) {
            UacRetargetingConditionRule.RuleType.ALL -> ERetargetingConditionRuleType.RCRT_ALL
            UacRetargetingConditionRule.RuleType.NOT -> ERetargetingConditionRuleType.RCRT_NOT
            UacRetargetingConditionRule.RuleType.OR -> ERetargetingConditionRuleType.RCRT_OR
        }
    }

    private fun UacRetargetingConditionRuleGoalType.toERetargetingConditionRuleGoalType(): ERetargetingConditionRuleGoalType {
        return when (this) {
            UacRetargetingConditionRuleGoalType.MOBILE -> ERetargetingConditionRuleGoalType.RCRGT_MOBILE
            UacRetargetingConditionRuleGoalType.LAL -> ERetargetingConditionRuleGoalType.RCRGT_LAL
            UacRetargetingConditionRuleGoalType.AUDIENCE -> ERetargetingConditionRuleGoalType.RCRGT_AUDIENCE
            UacRetargetingConditionRuleGoalType.LAL_AUDIENCE -> ERetargetingConditionRuleGoalType.RCRGT_LAL_AUDIENCE
            UacRetargetingConditionRuleGoalType.SEGMENT -> ERetargetingConditionRuleGoalType.RCRGT_SEGMENT
            UacRetargetingConditionRuleGoalType.LAL_SEGMENT -> ERetargetingConditionRuleGoalType.RCRGT_LAL_SEGMENT
            UacRetargetingConditionRuleGoalType.INTERESTS -> ERetargetingConditionRuleGoalType.RCRGT_INTERESTS
            UacRetargetingConditionRuleGoalType.HOST -> ERetargetingConditionRuleGoalType.RCRGT_HOST
        }
    }

    private fun ERetargetingConditionRuleGoalType.toUacRetargetingConditionRuleGoalType(): UacRetargetingConditionRuleGoalType? {
        return when (this) {
            ERetargetingConditionRuleGoalType.RCRGT_MOBILE -> UacRetargetingConditionRuleGoalType.MOBILE
            ERetargetingConditionRuleGoalType.RCRGT_LAL -> UacRetargetingConditionRuleGoalType.LAL
            ERetargetingConditionRuleGoalType.RCRGT_AUDIENCE -> UacRetargetingConditionRuleGoalType.AUDIENCE
            ERetargetingConditionRuleGoalType.RCRGT_LAL_AUDIENCE -> UacRetargetingConditionRuleGoalType.LAL_AUDIENCE
            ERetargetingConditionRuleGoalType.RCRGT_SEGMENT -> UacRetargetingConditionRuleGoalType.SEGMENT
            ERetargetingConditionRuleGoalType.RCRGT_LAL_SEGMENT -> UacRetargetingConditionRuleGoalType.LAL_SEGMENT
            ERetargetingConditionRuleGoalType.RCRGT_INTERESTS -> UacRetargetingConditionRuleGoalType.INTERESTS
            ERetargetingConditionRuleGoalType.RCRGT_HOST -> UacRetargetingConditionRuleGoalType.HOST
            else -> null
        }
    }

    private fun ECryptaInterestType.toCryptaInterestType(): CryptaInterestType? {
        return when (this) {
            ECryptaInterestType.CIT_SHORT_TERM -> CryptaInterestType.short_term
            ECryptaInterestType.CIT_LONG_TERM -> CryptaInterestType.long_term
            ECryptaInterestType.CIT_ALL -> CryptaInterestType.all
            ECryptaInterestType.CIT_NOT_SPECIFIED -> null
        }
    }

    private fun CryptaInterestType.toECryptaInterestType(): ECryptaInterestType {
        return when (this) {
            CryptaInterestType.short_term -> ECryptaInterestType.CIT_SHORT_TERM
            CryptaInterestType.long_term -> ECryptaInterestType.CIT_LONG_TERM
            CryptaInterestType.all -> ECryptaInterestType.CIT_ALL
        }
    }

    fun TCampaignBrief.TFeedFilter.toUacFeedFilter(): UacFeedFilter {
        return UacFeedFilter(
            conditions = conditionsList.map {
                UacFeedFilterCondition(
                    field = it.field,
                    operator = it.operator.toUacFeedFilterOperator()!!,
                    value = it.value,
                    values = it.value.toFeedFilterValues()
                )
            }
        )
    }

    fun TCampaignBrief.TRelevanceMatch.toRelevanceMatch() = UacRelevanceMatch(
        active = this.isActive,
        categories = this.categoriesList
            .mapNotNull { toRelevanceMatchCategory(it) }
            .toSet(),
    )

    private fun toRelevanceMatchCategory(value: Int): UacRelevanceMatchCategory? {
        // https://a.yandex-team.ru/arc/trunk/arcadia/grut/libs/proto/objects/campaign.proto?rev=r9238611#L412
        return UacRelevanceMatchCategory.fromId(value)
    }

    fun toTRelevanceMatch(
        relevanceMatch: UacRelevanceMatch
    ) = TCampaignBrief.TRelevanceMatch.newBuilder().apply {
        isActive = relevanceMatch.active
        addAllCategories(relevanceMatch.categories
            .map { toERelevanceMatchCategory(it) }
            .toSortedSet()
        )
    }.build()

    private fun toERelevanceMatchCategory(relevanceMatchCategory: UacRelevanceMatchCategory): Int {
        // https://a.yandex-team.ru/arc/trunk/arcadia/grut/libs/proto/objects/campaign.proto?rev=r9238611#L412
        return relevanceMatchCategory.id
    }

    private fun buildFeedFilter(uacFeedFilter: UacFeedFilter): TCampaignBrief.TFeedFilter {
        return TCampaignBrief.TFeedFilter.newBuilder().apply {
            addAllConditions(uacFeedFilter.conditions.map {
                TCampaignBrief.TFeedFilter.TCondition.newBuilder().apply {
                    field = it.field
                    operator = it.operator.toEFeedFilterOperator()
                    value = it.values?.toFeedFilterValue() ?: it.value
                }.build()
            })
        }.build()
    }

    private fun buildEcomData(campaign: UacYdbCampaign): TCampaignBrief.TEcomData? {
        if (campaign.isEcom == null || !campaign.isEcom) {
            return null
        }
        return TCampaignBrief.TEcomData.newBuilder().apply {
            campaign.feedId?.let { feedId = it }
            campaign.feedFilters?.let { feedFilters ->
                addAllFeedFilters(feedFilters.map { buildFeedFilter(it) })
            }
        }.build()
    }

    fun buildCpmAsset(cpmAsset: UacCpmAsset): TCpmAsset {
        return TCpmAsset.newBuilder().apply {
            cpmAsset.title?.let { title = it }
            cpmAsset.titleExtension?.let { titleExtension = it }
            cpmAsset.body?.let { body = it }
            cpmAsset.button?.let { button ->
                buttonBuilder.apply {
                    action = button.action.name
                    button.customText?.let { customText = it }
                    href = button.href
                }
            }
            cpmAsset.logoImageHash?.let { logoImageHash = cpmAsset.logoImageHash }
            cpmAsset.logoImage?.let { logoImage ->
                logoImageBuilder.apply {
                    hash = logoImage.hash
                    height = logoImage.height
                    width = logoImage.width
                    name = logoImage.name
                    path = logoImage.path
                    namespace = logoImage.namespace
                    mdsGroupId = logoImage.mdsGroupId
                }
            }
            cpmAsset.measurers?.let { addAllBannerMeasurers(it.map { measurer -> buildUacMeasurer(measurer) }) }
            cpmAsset.pixels?.let { addAllPixels(it) }
            cpmAsset.bannerHref?.let { bannerHref = it }
        }.build()
    }

    private fun buildUacMeasurer(uacMeasurer: UacMeasurer): TCampaignBrief.TCpmBannerMeasurers {
        return TCampaignBrief.TCpmBannerMeasurers.newBuilder().apply {
            measurerType = uacMeasurer.measurerType.name
            params = uacMeasurer.params
        }.build()
    }

    private fun toUacMeasurer(tCpmBannerMeasurers: TCampaignBrief.TCpmBannerMeasurers): UacMeasurer {
        return UacMeasurer(
            UacBannerMeasurerSystem.valueOf(tCpmBannerMeasurers.measurerType),
            tCpmBannerMeasurers.params
        )
    }

    fun TCpmAsset.toCpmAsset(): UacCpmAsset {
        return UacCpmAsset(
            title = if (hasTitle()) title else null,
            titleExtension = if (hasTitleExtension()) titleExtension else null,
            body = if (hasBody()) body else null,
            button = if (hasButton()) {
                CpmAssetButton(
                    action = UacButtonAction.valueOf(button.action),
                    customText = button.customText,
                    href = button.href,
                )
            } else null,
            logoImageHash = if (hasLogoImageHash()) logoImageHash else null,
            logoImage = if (hasLogoImage()) {
                CpmAssetLogo(
                    hash = logoImage.hash,
                    height = logoImage.height,
                    width = logoImage.width,
                    name = logoImage.name,
                    path = logoImage.path,
                    namespace = logoImage.namespace,
                    mdsGroupId = logoImage.mdsGroupId,
                )
            } else null,
            measurers = if (bannerMeasurersCount > 0) bannerMeasurersList.map { toUacMeasurer(it) } else null,
            pixels = if (pixelsCount > 0) pixelsList else null,
            bannerHref = if (hasBannerHref()) bannerHref else null,
        )
    }

    fun buildRetargetingConditionRule(rule: UacRetargetingConditionRule): TRetargetingConditionRule {
        return TRetargetingConditionRule.newBuilder().apply {
            type = rule.type.toERetargetingConditionRuleType()
            rule.interestType?.let { interestType = it.toECryptaInterestType() }
            addAllGoals(rule.goals.map { goal ->
                TRetargetingConditionRule.TRetargetingConditionRuleGoal.newBuilder().apply {
                    id = goal.id!!
                    goal.time?.let { time = it }
                    goal.name?.let { name = it }
                    goal.type?.let { type = it.toERetargetingConditionRuleGoalType() }
                    goal.description?.let { description = it }
                }.build()
            })
        }.build()
    }

    private fun buildCpmData(campaign: UacYdbCampaign): TCpmData? {
        if (campaign.advType != AdvType.CPM_BANNER) {
            return null
        }

        return TCpmData.newBuilder().apply {
            campaign.brandSurveyId?.let { brandSurveyId = it }
            campaign.retargetingCondition?.let { retargetingCondition ->
                retargetingConditionBuilder.apply {
                    retargetingCondition.name?.let { name = it }
                    retargetingCondition.id?.let { id = it }
                    addAllConditionRules(retargetingCondition.conditionRules.map { buildRetargetingConditionRule(it) })
                }
            }
            campaign.showsFrequencyLimit?.let { showsFreqLimit ->
                showsFrequencyLimitBuilder.apply {
                    impressionRateCount = showsFreqLimit.impressionRateCount
                    impressionRateIntervalDays = showsFreqLimit.impressionRateIntervalDays
                }
            }
            campaign.cpmAssets?.let { assets ->
                putAllCpmAssets(assets.mapValues { buildCpmAsset(it.value) })
            }

            campaign.campaignMeasurers?.let { measurers ->
                addAllCampaignMeasurers(
                    measurers.map { buildCampaignMeasurer(it) }
                )
            }

            campaign.uacBrandsafety?.let {
                brandSafety = buildBrandSafety(it)
            }

            campaign.uacDisabledPlaces?.let {
                disabledPlaces = buildDisabledPlaces(it);
            }

            campaign.searchLift?.let {
                searchLift = buildSearchLift(it)
            }
        }.build()
    }

    fun buildDisabledPlaces(uacDisabledPlaces: UacDisabledPlaces): TCampaignBrief.TDisabledPlaces {
        return TCampaignBrief.TDisabledPlaces.newBuilder().apply {
            uacDisabledPlaces.disabledPlaces?.let {
                addAllDisabledPlaces(uacDisabledPlaces.disabledPlaces)
            }
            uacDisabledPlaces.disabledIps?.let {
                addAllDisabledIps(uacDisabledPlaces.disabledIps)
            }
            uacDisabledPlaces.disabledVideoAdsPlaces?.let {
                addAllDisabledVideoAdsPlaces(uacDisabledPlaces.disabledVideoAdsPlaces)
            }
            uacDisabledPlaces.disallowedPageIds?.let {
                addAllDisallowedPageIds(uacDisabledPlaces.disallowedPageIds)
            }
        }.build()
    }

    fun buildBrandSafety(uacBrandsafety: UacBrandsafety): TCampaignBrief.TBrandSafety {
        return TCampaignBrief.TBrandSafety.newBuilder().apply {
            isEnabled = uacBrandsafety.isEnabled
            addAllAdditionalCategories(uacBrandsafety.additionalCategories)
        }.build()
    }

    fun buildSearchLift(uacSearchLift: UacSearchLift): TCampaignBrief.TSearchLift {
        return TCampaignBrief.TSearchLift.newBuilder().apply {
            uacSearchLift.brands?.let {
                addAllBrands(uacSearchLift.brands)
            }
            uacSearchLift.searchObjects?.let {
                addAllSearchObjects(uacSearchLift.searchObjects)
            }
        }.build()
    }

    fun buildCampaignMeasurer(uacCampaignMeasurer: UacCampaignMeasurer): TCampaignBrief.TCampaignMeasurer {
        return TCampaignBrief.TCampaignMeasurer.newBuilder().apply {
            measurerType = uacCampaignMeasurer.measurerType.name
            params = uacCampaignMeasurer.params
        }.build()
    }

    fun TCampaignBrief.TBrandSafety.toUacBrandSafety() = UacBrandsafety(isEnabled, additionalCategoriesList)

    fun TCampaignBrief.TDisabledPlaces.toUacDisabledPlaces() =
        UacDisabledPlaces(disabledPlacesList, disabledVideoAdsPlacesList, disabledIpsList, disallowedPageIdsList)

    fun TCampaignBrief.TSearchLift.toUacSearchLift() = UacSearchLift(brandsList, searchObjectsList)

    fun buildRetargetingCondition(retargetingCondition: UacRetargetingCondition): TRetargetingCondition {
        return TRetargetingCondition.newBuilder().apply {
            retargetingCondition.name?.let { name = it }
            retargetingCondition.id?.let { id = it }
            addAllConditionRules(retargetingCondition.conditionRules.map { buildRetargetingConditionRule(it) })
        }.build()
    }

    fun TRetargetingCondition.toRetargetingCondition(): UacRetargetingCondition? {
        if (conditionRulesCount == 0 && (!hasName() || name.isEmpty()) && (!hasId() || id == 0L)) {
            return null
        }
        return UacRetargetingCondition(
            conditionRules = conditionRulesList.map { rule ->
                UacRetargetingConditionRule(
                    type = rule.type.toRetargetingConditionRuleType()!!,
                    interestType = if (rule.hasInterestType()) rule.interestType.toCryptaInterestType() else null,
                    goals = rule.goalsList.map { goal ->
                        UacRetargetingConditionRuleGoal(
                            id = goal.id,
                            time = if (goal.hasTime()) goal.time else null,
                            name = if (goal.hasName()) goal.name else null,
                            type = if (goal.hasType()) goal.type.toUacRetargetingConditionRuleGoalType() else null,
                            description = if (goal.hasDescription()) goal.description else null,
                        )
                    }
                )
            },
            name = if (hasName()) name else null,
            id = if (hasId()) id else null,
        )
    }

    fun TShowsFrequencyLimit.toShowsFrequencyLimit() = UacShowsFrequencyLimit(
        impressionRateCount = impressionRateCount,
        impressionRateIntervalDays = impressionRateIntervalDays,
    )

    private fun buildAdjustment(uacAdjustment: UacAdjustment): TAdjustment {
        return TAdjustment.newBuilder().apply {
            uacAdjustment.age?.let { age = it.toEAgePoint() }
            uacAdjustment.gender?.let { gender = it.toEGender() }
            uacAdjustment.region?.let { region = it }
            uacAdjustment.percent.let { percent = it }
            uacAdjustment.retargetingConditionId?.let { retargetingConditionId = it }
        }.build()
    }

    fun List<UacAdjustment>.toTAdjustments(): List<TAdjustment> = map { buildAdjustment(it) }

    fun TAdjustment.toUacAdjustment() = UacAdjustment(
        age = if (hasAge()) age.toAgePoint() else null,
        gender = if (hasGender()) gender.toGender() else null,
        region = if (hasRegion()) region else null,
        percent = percent,
        retargetingConditionId = if (hasRetargetingConditionId()) retargetingConditionId else null,
        regionName = null,
    )

    fun getRetargetingCondition(brief: TCampaignBrief): UacRetargetingCondition? {
        return if (brief.cpmData.hasRetargetingCondition()) {
            brief.cpmData.retargetingCondition.toRetargetingCondition()
        } else if (brief.hasRetargetingCondition()) {
            brief.retargetingCondition.toRetargetingCondition()
        } else {
            null
        }
    }

    private fun getUacCampaign(
        campaign: TCampaign,
        advType: AdvType? = null,
        campaignId: String? = null,
        clientId: String? = null,
        targetStatus: TargetStatus? = null,
    ): UacYdbCampaign {
        val spec = campaign.spec
        val brief = spec.campaignBrief
        val data = spec.campaignData
        val localAdvType = advType ?: campaign.meta.campaignType.toAdvType()!!
        val localCampaignId = campaignId ?: campaign.meta.id.toIdString()
        val localClientId = clientId ?: campaign.meta.clientId.toIdString()
        val localTargetStatus = targetStatus ?: brief.targetStatus.toTargetStatus()!!

        return UacYdbCampaign(
            id = localCampaignId,
            name = data.name,
            storeUrl = brief.targetHref.href,
            appId = if (brief.hasAppId()) brief.appId else null,
            regions = brief.regionsList,
            minusRegions = if (brief.minusRegionsCount > 0) brief.minusRegionsList else null,
            trackingUrl = if (brief.targetHref.hasTrackingUrl()) brief.targetHref.trackingUrl else null,
            impressionUrl = if (brief.targetHref.hasImpressionUrl()) brief.targetHref.impressionUrl else null,
            createdAt = DateTimeUtils.fromEpochSeconds(campaign.meta.creationTime / 1_000_000),
            updatedAt = DateTimeUtils.fromEpochSeconds(spec.updateTime.toLong()),
            startedAt = if (spec.hasStartTime()) {
                DateTimeUtils.fromEpochSeconds(spec.startTime.toLong())
            } else {
                null
            },
            targetId = if (brief.strategy.hasMobileAppCampaignTargetType()) {
                brief.strategy.mobileAppCampaignTargetType.toTargetType()
            } else {
                null
            },
            cpa = if (brief.strategy.hasCpa()) moneyFromDb(brief.strategy.cpa) else null,
            crr = if (brief.strategy.hasCrr()) brief.strategy.crr else null,
            weekLimit = moneyFromDb(brief.strategy.weekLimit),
            account = localClientId,
            options = if (brief.strategy.hasLimitPeriod()) {
                brief.strategy.limitPeriod.toLimitPeriodType()?.let { UacCampaignOptions(it) }
            } else {
                null
            },
            skadNetworkEnabled = if (brief.hasSkadNetworkEnabled()) brief.skadNetworkEnabled else null,
            adultContentEnabled = if (brief.hasAdultContentEnabled()) brief.adultContentEnabled else null,
            targetStatus = localTargetStatus,
            contentFlags = if (brief.hasContentFlags()) brief.contentFlags.toContentFlags() else null,
            advType = localAdvType,
            hyperGeoId = if (brief.hasHypergeoId()) brief.hypergeoId else null,
            keywords = if (brief.keywordsCount > 0) brief.keywordsList else null,
            minusKeywords = if (brief.minusKeywordsCount > 0) brief.minusKeywordsList else null,
            socdem = if (brief.hasSocdem()) brief.socdem.toSocdem() else null,
            deviceTypes = if (brief.deviceTypeCount > 0) {
                brief.deviceTypeList.toDeviceTypes()
            } else {
                getDefaultDeviceTypes(localAdvType == AdvType.CPM_BANNER)
            },
            inventoryTypes = if (brief.inventoryTypeCount > 0) {
                brief.inventoryTypeList.toInventoryTypes()
            } else {
                getDefaultInventoryTypes(localAdvType)
            },
            goals = if (brief.strategy.goalsCount > 0) brief.strategy.goalsList.toUacGoals() else null,
            counters = if (data.countersCount > 0) data.countersList else null,
            permalinkId = if (brief.hasPermalinkId()) brief.permalinkId else null,
            phoneId = if (brief.hasPhoneId()) brief.phoneId else null,
            calltrackingSettingsId = if (brief.hasCalltrackingSettingsId()) brief.calltrackingSettingsId else null,
            timeTarget = if (brief.hasTimeTarget()) brief.timeTarget.toTimeTarget() else null,
            strategy = brief.strategy.toUacStrategy(),
            retargetingCondition = getRetargetingCondition(brief),
            videosAreNonSkippable = if (brief.hasVideosAreNonSkippable()) brief.videosAreNonSkippable else null,
            zenPublisherId = if (brief.hasZenPublisherId()) brief.zenPublisherId else null,
            brandSurveyId = if (brief.cpmData.hasBrandSurveyId()) brief.cpmData.brandSurveyId else null,
            assetLinks = readAssetLinks(
                spec.briefAssetLinks,
                spec.briefAssetLinksStatuses,
                localCampaignId
            ),
            directCampaignStatus = if (spec.hasStatus()) {
                spec.status.toDirectCampaignStatus()
            } else {
                null
            },
            showsFrequencyLimit = if (brief.cpmData.hasShowsFrequencyLimit()) {
                brief.cpmData.showsFrequencyLimit.toShowsFrequencyLimit()
            } else {
                null
            },
            briefSynced = if (brief.hasBriefSynced()) brief.briefSynced else null,
            strategyPlatform = if (brief.strategy.hasPlatform()) {
                brief.strategy.platform.toUacStrategyPlatform()
            } else {
                null
            },
            isEcom = brief.hasEcom(),
            feedId = if (brief.ecom.hasFeedId()) brief.ecom.feedId else null,
            feedFilters = if (brief.ecom.feedFiltersCount > 0) {
                brief.ecom.feedFiltersList.map { it.toUacFeedFilter() }
            } else {
                null
            },
            cpmAssets = if (brief.cpmData.cpmAssetsMap.isNotEmpty()) {
                brief.cpmData.cpmAssetsMap.mapValues { it.value.toCpmAsset() }
            } else {
                null
            },
            campaignMeasurers = if (brief.cpmData.campaignMeasurersList.isNotEmpty()) {
                brief.cpmData.campaignMeasurersList.map {
                    UacCampaignMeasurer(
                        UacCampaignMeasurerSystem.valueOf(it.measurerType),
                        it.params
                    )
                }
            } else {
                null
            },
            uacBrandsafety = if (brief.cpmData.hasBrandSafety()) {
                brief.cpmData.brandSafety?.toUacBrandSafety()
            } else {
                null
            },
            uacDisabledPlaces = if (brief.cpmData.hasDisabledPlaces()) {
                brief.cpmData.disabledPlaces?.toUacDisabledPlaces()
            } else if (brief.hasDisabledPlaces()) {
                brief.disabledPlaces?.toUacDisabledPlaces()
            } else {
                null
            },
            trackingParams = if (brief.hasTrackingParams()) brief.trackingParams else null,
            recommendationsManagementEnabled = if (brief.hasRecommendationsManagementEnabled()) {
                brief.recommendationsManagementEnabled
            } else {
                null
            },
            priceRecommendationsManagementEnabled = if (brief.hasPriceRecommendationsManagementEnabled()) {
                brief.priceRecommendationsManagementEnabled
            } else {
                null
            },
            relevanceMatch = if (brief.hasRelevanceMatch()) {
                brief.relevanceMatch.toRelevanceMatch()
            } else {
                null
            },
            adjustments = if (brief.adjustmentsCount > 0) {
                brief.adjustmentsList.map { it.toUacAdjustment() }
            } else {
                null
            },
            showTitleAndBody = if (brief.hasShowTitleAndBody()) brief.showTitleAndBody else null,
            bizLandingId = if (spec.hasBizLandingId() && spec.bizLandingId > 0) spec.bizLandingId else null,
            audienceSegmentsSynchronized = if (brief.hasAudienceSegmentsSynchronized()) brief.audienceSegmentsSynchronized else null,
            searchLift = if (brief.cpmData.hasSearchLift()) {
                brief.cpmData.searchLift?.toUacSearchLift()
            } else {
                null
            },
        )
    }
}

class ProtoEnumNumberHolder {
    private val numbersDataByEnumFullName: ConcurrentMap<String, Map<String, Int>> = ConcurrentHashMap()

    private fun getOrCreateNumbersData(enumTypeDescriptor: Descriptors.EnumDescriptor): Map<String, Int> {
        return numbersDataByEnumFullName.getOrPut(enumTypeDescriptor.fullName) {
            val numberByName = HashMap<String, Int>()
            for (enumValueDescriptor in enumTypeDescriptor.values) {
                for ((k, v) in enumValueDescriptor.options.allFields) {
                    if (k.fullName == ENUM_VALUE_NAME_OPTION) {
                        numberByName[v as String] = enumValueDescriptor.number
                    }
                }
            }
            numberByName
        }
    }

    fun getProtoEnumValueNumber(valueName: String, enumTypeDescriptor: Descriptors.EnumDescriptor): Int? {
        return getOrCreateNumbersData(enumTypeDescriptor)[valueName]
    }
}

private const val ENUM_VALUE_NAME_OPTION = "NYT.NYson.NProto.enum_value_name"

fun getProtoEnumValueName(enumValueDescriptor: Descriptors.EnumValueDescriptor): String? {
    for ((key, value) in enumValueDescriptor.options.allFields) {
        if (key.fullName == ENUM_VALUE_NAME_OPTION) {
            return value as String
        }
    }
    return null
}

fun <E : ProtocolMessageEnum> getProtoEnumValueName(value: E): String {
    return getProtoEnumValueName(value.valueDescriptor)
        ?: throw RuntimeException(
            "Enum value \"${value.valueDescriptor.fullName}\" does not have \"$ENUM_VALUE_NAME_OPTION\" option defined"
        )
}

val PROTO_ENUM_NUMBER_HOLDER = ProtoEnumNumberHolder()


