package ru.yandex.direct.core.entity.uac.repository.mysql

import java.time.LocalDateTime
import org.jooq.Condition
import org.jooq.DSLContext
import org.jooq.OrderField
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.util.RepositoryUtils
import ru.yandex.direct.core.entity.uac.model.EcomDomain
import ru.yandex.direct.dbschema.ppcdict.tables.EcomDomains.ECOM_DOMAINS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import ru.yandex.direct.jooqmapperhelper.UpdateHelper
import ru.yandex.direct.model.AppliedChanges

@Repository
class EcomDomainsRepository @Autowired constructor(private val dslContextProvider: DslContextProvider) {
    companion object {
        private val MAPPER: JooqMapperWithSupplier<EcomDomain> = JooqMapperWithSupplierBuilder.builder(::EcomDomain)
            .map(property(EcomDomain.ID, ECOM_DOMAINS.DOMAIN_ID))
            .map(property(EcomDomain.DOMAIN, ECOM_DOMAINS.DOMAIN))
            .map(property(EcomDomain.SCHEMA, ECOM_DOMAINS.SCHEMA))
            .map(property(EcomDomain.OFFERS_COUNT, ECOM_DOMAINS.OFFERS_COUNT))
            .map(convertibleProperty(EcomDomain.IS_PERMANENT, ECOM_DOMAINS.IS_PERMANENT,
                RepositoryUtils::booleanFromLong, RepositoryUtils::nullSafeBooleanToLong))
            .map(property(EcomDomain.MARKET_BUSINESS_ID, ECOM_DOMAINS.MARKET_BUSINESS_ID))
            .map(property(EcomDomain.MARKET_SHOP_ID, ECOM_DOMAINS.MARKET_SHOP_ID))
            .map(property(EcomDomain.MARKET_FEED_ID, ECOM_DOMAINS.MARKET_FEED_ID))
            .map(property(EcomDomain.PREVIEW_OFFERS, ECOM_DOMAINS.PREVIEW_OFFERS))
            .map(property(EcomDomain.LAST_MODIFIED_PREVIEW, ECOM_DOMAINS.LAST_MODIFIED_PREVIEW))
            .build()
    }

    fun get(dslContext: DSLContext,
            condition: Condition,
            limit: Int?,
            orderBy: OrderField<out Any>? = null): List<EcomDomain> {
        val selectConditionStep = dslContext
            .select(MAPPER.fieldsToRead)
            .from(ECOM_DOMAINS)
            .where(condition)

        if (orderBy != null) {
            selectConditionStep.orderBy(orderBy)
        }

        if (limit != null) {
            selectConditionStep.limit(limit)
        }

        return selectConditionStep.fetch(MAPPER::fromDb)
    }

    fun getByDomainsAndMinMaxOffers(domains: Collection<String>, minOffersCount: Long, maxOffersCount: Long): List<EcomDomain> {
        val condition = ECOM_DOMAINS.DOMAIN.`in`(domains)
            .and(ECOM_DOMAINS.OFFERS_COUNT.between(minOffersCount, maxOffersCount))
        val dslContext = dslContextProvider.ppcdict()
        return get(dslContext, condition, null)
    }

    fun getByDomains(domains: Collection<String>): List<EcomDomain> {
        val condition = ECOM_DOMAINS.DOMAIN.`in`(domains)
        val dslContext = dslContextProvider.ppcdict()
        return get(dslContext, condition, null)
    }

    fun getByBusinessIds(ids: Collection<Long>): List<EcomDomain> {
        val condition = ECOM_DOMAINS.MARKET_BUSINESS_ID.`in`(ids)
        val dslContext = dslContextProvider.ppcdict()
        return get(dslContext, condition, null)
    }

    /**
     * Получить еком домены, для которых еще не были проставлены маркетные id, с условием, что количество оферов
     * у них больше указанного или домены зарегистрированы вручную (is_permanent)
     */
    fun getWithoutMarketIds(minOffersCount: Long, limit: Int? = null): List<EcomDomain> {
        val condition = notSentToMbiCondition(minOffersCount)
        val dslContext = dslContextProvider.ppcdict()
        return get(dslContext, condition, limit)
    }

    fun getEcomDomainsWithOutdatedPreviews(expirationHours: Long): List<EcomDomain> {
        val condition = ECOM_DOMAINS.LAST_MODIFIED_PREVIEW.lt(LocalDateTime.now().minusHours(expirationHours))
            .and(ECOM_DOMAINS.MARKET_BUSINESS_ID.isNotNull)
            .and(ECOM_DOMAINS.MARKET_SHOP_ID.isNotNull)
        return get(dslContextProvider.ppcdict(), condition, null, ECOM_DOMAINS.LAST_MODIFIED_PREVIEW.asc())
    }

    fun getWithPreviewIsNotNull(): List<EcomDomain> {
        val condition = ECOM_DOMAINS.PREVIEW_OFFERS.isNotNull
        val dslContext = dslContextProvider.ppcdict()
        return get(dslContext, condition, null)
    }

    fun getAllDomainsToIds(): Map<String, Long> {
        return dslContextProvider.ppcdict().select(ECOM_DOMAINS.DOMAIN_ID, ECOM_DOMAINS.DOMAIN)
            .from(ECOM_DOMAINS)
            .fetchMap(ECOM_DOMAINS.DOMAIN)
            .mapValues { it.value.value1() }
    }

    fun getSentDomainsToMBI(minId: Long, limit: Int): List<EcomDomain> {
        val condition = ECOM_DOMAINS.MARKET_FEED_ID.isNotNull
            .and(ECOM_DOMAINS.DOMAIN_ID.greaterOrEqual(minId))
        val dslContext = dslContextProvider.ppcdict()
        return get(dslContext, condition, limit, ECOM_DOMAINS.DOMAIN_ID)
    }

    fun getEcomDomainsTotalCount(): Int {
        return dslContextProvider.ppcdict().selectCount()
            .from(ECOM_DOMAINS)
            .fetchOne(0, Int::class.java)
    }

    fun getSentToMbiEcomDomainsCount(): Int {
        return dslContextProvider.ppcdict().selectCount()
            .from(ECOM_DOMAINS)
            .where(ECOM_DOMAINS.MARKET_FEED_ID.isNotNull)
            .fetchOne(0, Int::class.java)
    }

    fun getNotSentToMbiEcomDomainsCount(minOffersCount: Long): Int {
        return dslContextProvider.ppcdict().selectCount()
            .from(ECOM_DOMAINS)
            .where(notSentToMbiCondition(minOffersCount))
            .fetchOne(0, Int::class.java)
    }

    fun getEcomDomainsWithPreviewCount(): Int {
        return dslContextProvider.ppcdict().selectCount()
            .from(ECOM_DOMAINS)
            .where(ECOM_DOMAINS.PREVIEW_OFFERS.isNotNull)
            .fetchOne(0, Int::class.java)
    }

    fun getEcomDomainsWithoutUpdatingPreviewCount(daysLimit: Long): Int {
        return dslContextProvider.ppcdict().selectCount()
            .from(ECOM_DOMAINS)
            .where(ECOM_DOMAINS.PREVIEW_OFFERS.isNotNull
                .and(ECOM_DOMAINS.LAST_MODIFIED_PREVIEW.isNull
                    .or(ECOM_DOMAINS.LAST_MODIFIED_PREVIEW.lt(LocalDateTime.now().minusDays(daysLimit)))))
            .fetchOne(0, Int::class.java)
    }

    fun update(dslContext: DSLContext, appliedChanges: List<AppliedChanges<EcomDomain>>): Int {
        return UpdateHelper(dslContext, ECOM_DOMAINS.DOMAIN_ID)
            .processUpdateAll(MAPPER, appliedChanges)
            .execute()
    }

    fun update(appliedChanges: List<AppliedChanges<EcomDomain>>): Int {
        val dslContext = dslContextProvider.ppcdict()
        return update(dslContext, appliedChanges)
    }

    fun insert(ecomDomains: List<EcomDomain>): Int {
        return InsertHelper(dslContextProvider.ppcdict(), ECOM_DOMAINS)
            .addAll(MAPPER, ecomDomains)
            .onDuplicateKeyIgnore()
            .executeIfRecordsAdded()
    }

    fun delete(domainIds: Collection<Long>): Int {
        return dslContextProvider.ppcdict().deleteFrom(ECOM_DOMAINS)
            .where(ECOM_DOMAINS.DOMAIN_ID.`in`(domainIds))
            .execute()
    }

    fun deleteNotPermanent(domainIds: Collection<Long>): Int {
        return dslContextProvider.ppcdict().deleteFrom(ECOM_DOMAINS)
            .where(ECOM_DOMAINS.DOMAIN_ID.`in`(domainIds).and(ECOM_DOMAINS.IS_PERMANENT.eq(0)))
            .execute()
    }

    private fun notSentToMbiCondition(minOffersCount: Long): Condition {
        return ECOM_DOMAINS.MARKET_FEED_ID.isNull
            .and(ECOM_DOMAINS.OFFERS_COUNT.greaterOrEqual(minOffersCount)
                .or(ECOM_DOMAINS.IS_PERMANENT.eq(1)))
    }
}
