package ru.yandex.direct.core.entity.uac.repository.mysql

import org.jooq.Condition
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.util.RepositoryUtils
import ru.yandex.direct.core.entity.uac.model.EcomOfferCatalog
import ru.yandex.direct.dbschema.ppcdict.tables.EcomOfferCatalogs.ECOM_OFFER_CATALOGS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import ru.yandex.direct.jooqmapperhelper.UpdateHelper
import ru.yandex.direct.model.AppliedChanges

@Repository
class EcomOfferCatalogsRepository @Autowired constructor(private val dslContextProvider: DslContextProvider) {
    companion object {
        private val MAPPER: JooqMapperWithSupplier<EcomOfferCatalog> =
            JooqMapperWithSupplierBuilder.builder(::EcomOfferCatalog)
                .map(property(EcomOfferCatalog.ID, ECOM_OFFER_CATALOGS.CATALOG_ID))
                .map(property(EcomOfferCatalog.HOST, ECOM_OFFER_CATALOGS.HOST))
                .map(property(EcomOfferCatalog.URL, ECOM_OFFER_CATALOGS.URL_PATH))
                .map(property(EcomOfferCatalog.TITLE, ECOM_OFFER_CATALOGS.TITLE))
                .map(property(EcomOfferCatalog.CATALOG_PATH, ECOM_OFFER_CATALOGS.CATALOG_PATH))
                .map(property(EcomOfferCatalog.IMAGE_URL, ECOM_OFFER_CATALOGS.IMAGE_URL))
                .map(property(EcomOfferCatalog.VISITS_COUNT, ECOM_OFFER_CATALOGS.VISITS_COUNT))
                .map(
                    convertibleProperty(
                        EcomOfferCatalog.IS_PERMANENT, ECOM_OFFER_CATALOGS.IS_PERMANENT,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::nullSafeBooleanToLong
                    )
                )
                .build()
    }

    fun get(condition: Condition, limit: Int? = null): List<EcomOfferCatalog> {
        val selectConditionStep = dslContextProvider.ppcdict()
            .select(MAPPER.fieldsToRead)
            .from(ECOM_OFFER_CATALOGS)
            .where(condition)

        if (limit != null) {
            selectConditionStep.limit(limit)
        }

        return selectConditionStep.fetch(MAPPER::fromDb)
    }

    fun getByHosts(hosts: Collection<String>): List<EcomOfferCatalog> {
        val condition = ECOM_OFFER_CATALOGS.HOST.`in`(hosts)
        return get(condition)
    }

    fun update(appliedChanges: Collection<AppliedChanges<EcomOfferCatalog>>): Int {
        return UpdateHelper(dslContextProvider.ppcdict(), ECOM_OFFER_CATALOGS.CATALOG_ID)
            .processUpdateAll(MAPPER, appliedChanges)
            .execute()
    }

    fun insert(ecomDomains: Collection<EcomOfferCatalog>): Int {
        return InsertHelper(dslContextProvider.ppcdict(), ECOM_OFFER_CATALOGS)
            .addAll(MAPPER, ecomDomains)
            .onDuplicateKeyIgnore()
            .executeIfRecordsAdded()
    }

    fun delete(domainIds: Collection<Long>): Int {
        return dslContextProvider.ppcdict().deleteFrom(ECOM_OFFER_CATALOGS)
            .where(ECOM_OFFER_CATALOGS.CATALOG_ID.`in`(domainIds))
            .execute()
    }

    fun deleteNotPermanent(domainIds: Collection<Long>): Int {
        return dslContextProvider.ppcdict().deleteFrom(ECOM_OFFER_CATALOGS)
            .where(ECOM_OFFER_CATALOGS.CATALOG_ID.`in`(domainIds).and(ECOM_OFFER_CATALOGS.IS_PERMANENT.eq(0)))
            .execute()
    }
}
