package ru.yandex.direct.core.entity.uac.repository.mysql

import org.jooq.Condition
import org.jooq.DSLContext
import org.jooq.util.mysql.MySQLDSL
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.uac.model.ShopInShopBusiness
import ru.yandex.direct.core.entity.uac.model.Source
import ru.yandex.direct.dbschema.ppcdict.tables.ShopInShopBusinesses.SHOP_IN_SHOP_BUSINESSES
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property
import ru.yandex.direct.jooqmapperhelper.InsertHelper

@Repository
class ShopInShopBusinessesRepository @Autowired constructor(private val dslContextProvider: DslContextProvider) {
    companion object {
        private val MAPPER: JooqMapperWithSupplier<ShopInShopBusiness> =
            JooqMapperWithSupplierBuilder.builder(::ShopInShopBusiness)
                .map(
                    convertibleProperty(
                        ShopInShopBusiness.SOURCE, SHOP_IN_SHOP_BUSINESSES.SOURCE,
                        Source::fromSource, Source::toSource
                    )
                )
                .map(property(ShopInShopBusiness.BUSINESS_ID, SHOP_IN_SHOP_BUSINESSES.BUSINESS_ID))
                .map(property(ShopInShopBusiness.FEED_URL, SHOP_IN_SHOP_BUSINESSES.FEED_URL))
                .map(property(ShopInShopBusiness.COUNTER_ID, SHOP_IN_SHOP_BUSINESSES.COUNTER_ID))
                .build()
    }

    fun get(dslContext: DSLContext, condition: Condition, limit: Int? = null): List<ShopInShopBusiness> {
        val selectConditionStep = dslContext
            .select(MAPPER.fieldsToRead)
            .from(SHOP_IN_SHOP_BUSINESSES)
            .where(condition)

        if (limit != null) {
            selectConditionStep.limit(limit)
        }

        return selectConditionStep.fetch(MAPPER::fromDb)
    }

    fun getBySource(source: Source, limit: Int? = null): List<ShopInShopBusiness> {
        val dslContext = dslContextProvider.ppcdict()
        return get(dslContext, getBySourceCondition(source), limit)
    }

    fun getBySourceAndBusinessId(source: Source, businessId: Long): ShopInShopBusiness? {
        val dslContext = dslContextProvider.ppcdict()
        val condition = getBySourceCondition(source)
            .and(SHOP_IN_SHOP_BUSINESSES.BUSINESS_ID.eq(businessId))
        return get(dslContext, condition).firstOrNull()
    }

    fun getBusinessIdsBySource(source: Source): List<Long> {
        return dslContextProvider.ppcdict()
            .select(SHOP_IN_SHOP_BUSINESSES.BUSINESS_ID)
            .from(SHOP_IN_SHOP_BUSINESSES)
            .where(getBySourceCondition(source))
            .fetch(SHOP_IN_SHOP_BUSINESSES.BUSINESS_ID)
    }

    fun saveShopInShopBusinesses(shopInShopBusiness: List<ShopInShopBusiness>): Int {
        return InsertHelper(dslContextProvider.ppcdict(), SHOP_IN_SHOP_BUSINESSES)
            .addAll(MAPPER, shopInShopBusiness)
            .onDuplicateKeyUpdate()
            .set(SHOP_IN_SHOP_BUSINESSES.FEED_URL, MySQLDSL.values(SHOP_IN_SHOP_BUSINESSES.FEED_URL))
            .set(SHOP_IN_SHOP_BUSINESSES.COUNTER_ID, MySQLDSL.values(SHOP_IN_SHOP_BUSINESSES.COUNTER_ID))
            .executeIfRecordsAdded()
    }

    fun delete(source: Source, businessIds: Collection<Long>): Int {
        return dslContextProvider.ppcdict().deleteFrom(SHOP_IN_SHOP_BUSINESSES)
            .where(
                SHOP_IN_SHOP_BUSINESSES.BUSINESS_ID.`in`(businessIds)
                    .and(getBySourceCondition(source))
            )
            .execute()
    }

    private fun getBySourceCondition(source: Source): Condition {
        return SHOP_IN_SHOP_BUSINESSES.SOURCE.eq(Source.toSource(source))
    }
}
